package ru.yandex.search.yc.iam;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.google.rpc.DebugInfo;
import com.google.rpc.Status;
import io.grpc.ManagedChannel;
import io.grpc.StatusRuntimeException;
import io.grpc.internal.DnsNameResolverProvider;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NettyChannelBuilder;
import io.grpc.protobuf.StatusProto;
import io.grpc.stub.StreamObserver;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import org.apache.http.concurrent.FutureCallback;
import yandex.cloud.priv.servicecontrol.v1.AccessServiceGrpc;
import yandex.cloud.priv.servicecontrol.v1.AccessServiceGrpc.AccessServiceBlockingStub;
import yandex.cloud.priv.servicecontrol.v1.AccessServiceGrpc.AccessServiceStub;
import yandex.cloud.priv.servicecontrol.v1.AccessServiceOuterClass;
import yandex.cloud.priv.servicecontrol.v1.AccessServiceOuterClass.AuthorizeRequest;
import yandex.cloud.priv.servicecontrol.v1.AccessServiceOuterClass.AuthorizeResponse;
import yandex.cloud.priv.servicecontrol.v1.PR;

import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.yc.FilterableResource;
import ru.yandex.search.yc.YcSearchProxy;
import ru.yandex.search.yc.config.ImmutableGrpcHostConfig;

public class IamResourceFilter {
    public static final String RESOURCE_TYPE_CLOUD = "resource-manager.cloud";
    public static final String RESOURCE_TYPE_FOLDER = "resource-manager.folder";

    private static final String DEFAULT_PERMISSION = "get";
    //private final AsyncCloudAuthClient iamClient;
    private final PrefixedLogger logger;
    private final AccessServiceBlockingStub stub;
    private final AccessServiceStub asyncStub;

    public IamResourceFilter(final YcSearchProxy proxy) throws IOException {
        ImmutableGrpcHostConfig iamConfig = proxy.config().iamConfig();
        NettyChannelBuilder channelBuilder = NettyChannelBuilder
            .forAddress(iamConfig.host().getHostName(), iamConfig.host().getPort())
            .nameResolverFactory(new DnsNameResolverProvider());
        if (iamConfig.keepAliveTime() > 0) {
            channelBuilder.keepAliveTime(
                iamConfig.keepAliveTime(),
                TimeUnit.MILLISECONDS);
        }

        if (iamConfig.keepAliveTimeout() > 0) {
            channelBuilder.keepAliveTimeout(
                iamConfig.keepAliveTimeout(),
                TimeUnit.MILLISECONDS);
        }

        if (iamConfig.retries().count() > 0) {
            channelBuilder
                .enableRetry()
                .maxRetryAttempts(iamConfig.retries().count());
        }

        channelBuilder.keepAliveWithoutCalls(iamConfig.keepAliveWithoutCalls());
        channelBuilder.userAgent(iamConfig.userAgent());
        if (iamConfig.https().trustManagerFactory() != null) {
            channelBuilder.sslContext(
                GrpcSslContexts.forClient()
                    .trustManager(InsecureTrustManagerFactory.INSTANCE)
                    //.trustManager(iamConfig.https().trustManagerFactory())
                    .build());
        } else {
            channelBuilder.usePlaintext();
        }

        ManagedChannel channel = channelBuilder.build();
        stub = AccessServiceGrpc.newBlockingStub(channel);
        asyncStub = AccessServiceGrpc.newStub(channel);
        logger = proxy.logger();
    }

    public <T extends FilterableResource> void authorizeCloud(
        final PrefixedLogger logger,
        final String token,
        final T resource,
        final FutureCallback<ResourceWithResolution<T>> callback)
    {
        FilterResourceCallback<T> resourceCallback =
            new FilterResourceCallback<>(logger, resource, callback);
        String permission = resource.permission();
        if (permission == null || permission.isEmpty()) {
            permission = DEFAULT_PERMISSION;
        }
        authorize(
            logger,
            resource.cloudId(),
            RESOURCE_TYPE_CLOUD,
            permission,
            token,
            resourceCallback);
    }

    public <T extends FilterableResource> void authorizeCloud(
        final String token,
        final T resource,
        final FutureCallback<ResourceWithResolution<T>> callback)
    {
        this.authorizeCloud(logger, token, resource, callback);
    }


    public <T extends FilterableResource> void authorize(
        final PrefixedLogger logger,
        final String token,
        final List<T> resources,
        final FutureCallback<List<ResourceWithResolution<T>>> callback)
    {
        MultiFutureCallback<ResourceWithResolution<T>> mfcb =
            new MultiFutureCallback<>(
                callback);

        for (T item: resources) {
            authorizeFolder(
                logger,
                token,
                item,
                mfcb.newCallback());
        }
        mfcb.done();
    }

    public <T extends FilterableResource> void authorizeFolder(
        final PrefixedLogger logger,
        final String token,
        final T resource,
        final FutureCallback<ResourceWithResolution<T>> callback)
    {
        FilterResourceCallback<T> resourceCallback =
            new FilterResourceCallback<>(logger, resource, callback);
        String permission = resource.permission();
        if (permission == null || permission.isEmpty()) {
            permission = DEFAULT_PERMISSION;
        }

        authorize(
            logger,
            resource.folderId(),
            RESOURCE_TYPE_FOLDER,
            permission,
            token,
            resourceCallback);
    }

    public <T extends FilterableResource> void authorizeFolder(
        final String token,
        final T resource,
        final FutureCallback<ResourceWithResolution<T>> callback)
    {
        this.authorizeFolder(logger, token, resource, callback);
    }

    private void authorize(
        final PrefixedLogger logger,
        final String resourceId,
        final String resourceType,
        final String permission,
        final String token,
        final FilterResourceCallback<?> resourceCallback)
    {
        authorize(false, logger, resourceId, resourceType, permission, token, resourceCallback);
    }

    public void authorize(
        final boolean sync,
        final PrefixedLogger logger,
        final String resourceId,
        final String resourceType,
        final String permission,
        final String token,
        final FilterResourceCallback<?> resourceCallback)
    {
        AuthorizeRequest.Builder builder
            = AuthorizeRequest.newBuilder();

        PR.Resource resource =
            PR.Resource.newBuilder()
                .setId(resourceId)
                .setType(resourceType)
                .build();

        builder.setPermission(permission);
        builder.addResourcePath(resource);
        String requestWithoutToken = builder.toString();

        builder.setIamToken(token);
        AuthorizeRequest request = builder.build();

        if (!sync) {
            asyncStub.authorize(request, new IamStreamObserver(resourceCallback, logger));
        } else {
            try {
                AuthorizeResponse response = stub.authorize(request);
                logger.info(
                    "For request: " + requestWithoutToken
                        + " iam response " + String.valueOf(response));
                resourceCallback.apply(response.getSubject(), null);
            } catch (StatusRuntimeException sre) {
                IamError iamError;
                try {
                    iamError = exceptionToError(logger, sre);
                } catch (IOException ioe) {
                    iamError =
                        new IamError(
                            IamErrorType.UNKNOWN,
                            "Malformed Protocol " + ioe.getMessage(),
                            ioe);
                }

                resourceCallback.apply(null, iamError);
            } catch (Exception e) {
                IamError iamError =
                    new IamError(IamErrorType.UNKNOWN, "Grpc failed", e);

                logger.log(Level.WARNING, "Grpc failed", e);
                resourceCallback.apply((AccessServiceOuterClass.Subject) null, iamError);
            }
        }
    }

    public <T extends FilterableResource> void authorizePathes(
        final PrefixedLogger logger,
        final String token,
        final T resource,
        final FutureCallback<ResourceWithResolution<T>> callback)
    {
        FilterResourceCallback<T> resourceCallback =
            new FilterResourceCallback<>(logger, resource, callback);
        String permission = resource.permission();
        if (permission == null || permission.isEmpty()) {
            permission = DEFAULT_PERMISSION;
        }

        this.authorizePathes(false, logger,
                Collections.singleton(resource.resourcePath().get(resource.resourcePath().size() - 1)), permission, token, resourceCallback);
    }

    private void authorizePathes(
        final boolean sync,
        final PrefixedLogger logger,
        final Collection<PR.Resource> pathes,
        final String permission,
        final String token,
        final FilterResourceCallback<?> resourceCallback)
    {
        AuthorizeRequest.Builder builder
            = AuthorizeRequest.newBuilder();

        builder.addAllResourcePath(pathes);
        builder.setPermission(permission);
        //builder.addResourcePath(resource);
        String requestWithoutToken = builder.toString();

        builder.setIamToken(token);
        AuthorizeRequest request = builder.build();

        if (!sync) {
            asyncStub.authorize(request, new IamStreamObserver(resourceCallback, logger));
        } else {
            try {
                AuthorizeResponse response = stub.authorize(request);
                logger.info(
                    "For request: " + requestWithoutToken
                        + " iam response " + String.valueOf(response));
                resourceCallback.apply(response.getSubject(), null);
            } catch (StatusRuntimeException sre) {
                IamError iamError;
                try {
                    iamError = exceptionToError(logger, sre);
                } catch (IOException ioe) {
                    iamError =
                        new IamError(
                            IamErrorType.UNKNOWN,
                            "Malformed Protocol " + ioe.getMessage(),
                            ioe);
                }

                resourceCallback.apply(null, iamError);
            } catch (Exception e) {
                IamError iamError =
                    new IamError(IamErrorType.UNKNOWN, "Grpc failed", e);

                logger.log(Level.WARNING, "Grpc failed", e);
                resourceCallback.apply((AccessServiceOuterClass.Subject) null, iamError);
            }
        }
    }

    private class IamStreamObserver implements StreamObserver<AuthorizeResponse> {
        private final FilterResourceCallback<?> resourceCallback;
        private final PrefixedLogger logger;

        public IamStreamObserver(
            final FilterResourceCallback<?> resourceCallback,
            final PrefixedLogger logger)
        {
            this.resourceCallback = resourceCallback;
            this.logger = logger;
        }

        @Override
        public void onNext(final AuthorizeResponse value) {
            resourceCallback.apply(value.getSubject(), null);
        }

        @Override
        public void onError(final Throwable e) {
            IamError iamError;
            if (e instanceof StatusRuntimeException) {
                try {
                    iamError = exceptionToError(logger, (StatusRuntimeException) e);
                } catch (IOException ioe) {
                    logger.log(Level.WARNING, "Grpc failed", e);
                    iamError =
                        new IamError(
                            IamErrorType.UNKNOWN,
                            "Malformed Protocol " + ioe.getMessage(),
                            ioe);
                }
            } else {
                iamError =
                    new IamError(IamErrorType.UNKNOWN, "Grpc failed", e);

                logger.log(Level.WARNING, "Grpc failed", e);
            }


            resourceCallback.apply(null, iamError);
        }

        @Override
        public void onCompleted() {
        }
    }

    private IamError exceptionToError(
        final Logger logger,
        final StatusRuntimeException ex)
        throws IOException
    {
        logger.log(Level.WARNING, "Status exception", ex);
        String internalDetails = getInternalDetails(ex);
        switch (ex.getStatus().getCode()) {
            case INVALID_ARGUMENT:
                return new IamError(IamErrorType.BAD_REQUEST, internalDetails, ex);
            case CANCELLED:
                return new IamError(IamErrorType.SERVER_ERROR, internalDetails, ex);
            case DEADLINE_EXCEEDED:
                return new IamError(IamErrorType.TIMEOUT, internalDetails, ex);
            case PERMISSION_DENIED:
                return new IamError(
                    IamErrorType.PERMISSION_DENIED,
                    internalDetails,
                    ex,
                    getSubjectDetails(ex));
            case UNIMPLEMENTED:
                return new IamError(IamErrorType.BAD_REQUEST, internalDetails, ex);
            case INTERNAL:
                return new IamError(IamErrorType.SERVER_ERROR, internalDetails, ex);
            case UNAVAILABLE:
                return new IamError(IamErrorType.SERVER_ERROR, internalDetails, ex);
            case UNAUTHENTICATED:
                return new IamError(IamErrorType.BAD_REQUEST, internalDetails, ex);
            default:
                return new IamError(IamErrorType.UNKNOWN, internalDetails, ex);
        }
    }

    private static AccessServiceOuterClass.Subject getSubjectDetails(
        final Throwable throwable)
        throws IOException
    {
        return unpackDetails(throwable, AccessServiceOuterClass.Subject.class);
    }

    private static String getInternalDetails(final Throwable throwable) throws IOException {
        DebugInfo info = unpackDetails(throwable, DebugInfo.class);
        return info == null ? null : info.getDetail();
    }

    private static <T extends Message> T unpackDetails(
        final Throwable throwable,
        final Class<T> clazz)
        throws IOException
    {
        Status status = StatusProto.fromThrowable(throwable);
        if (status != null) {
            List<Any> detailsList = status.getDetailsList();
            if (detailsList != null) {
                for (Any any : detailsList) {
                    if (any.is(clazz)) {
                        try {
                            return any.unpack(clazz);
                        } catch (InvalidProtocolBufferException ex) {
                            throw new IOException("Malformed protocol", ex);
                        }
                    }
                }
            }
        }
        return null;
    }

    private static class FilterResourceCallback<T extends FilterableResource>
        implements BiFunction<AccessServiceOuterClass.Subject, IamError, ResourceWithResolution<T>>
    {
        private final T resource;
        private final PrefixedLogger logger;
        private final FutureCallback<ResourceWithResolution<T>> callback;

        public FilterResourceCallback(
            final PrefixedLogger logger,
            final T resource,
            final FutureCallback<ResourceWithResolution<T>> callback)
        {
            this.resource = resource;
            this.callback = callback;
            this.logger = logger;
        }

        @Override
        public ResourceWithResolution<T> apply(
            final AccessServiceOuterClass.Subject subject,
            final IamError iamError)
        {
            ResourceWithResolution<T> result;
            if (iamError != null) {
                result = new ResourceWithResolution<>(
                    resource,
                    AuthorizationResolution.REJECT_IAM);

                if (iamError.errorType() == IamErrorType.PERMISSION_DENIED) {
                    StringBuilder log = new StringBuilder();
                    log.append("For resource id ");
                    log.append(resource.resourceId());
                    log.append(" and permissions ");
                    log.append(resource.permission());
                    log.append(" permission denied ");
                    log.append(String.valueOf(iamError.subject()));
                    log.append(iamError.details());
                    logger.log(Level.INFO, log.toString(), iamError.exception);

                    callback.completed(result);
                } else {
                    callback.failed(
                        new Exception("Iam exception " + iamError.toString(), iamError.exception));
                }
            } else {
                // no exception and subject not null,
                // supposing that rights are ok

                if (subject == null) {
                    logger.warning(
                        "For resource id " + resource.resourceId()
                            + " and permissions " + resource.permission() + " subject null");
                }

                result = new ResourceWithResolution<>(resource, AuthorizationResolution.ALLOW);
                callback.completed(result);
            }

            return result;
        }
    }

    private enum IamErrorType {
        UNKNOWN,
        BAD_REQUEST,
        PERMISSION_DENIED,
        SERVER_ERROR,
        TIMEOUT
    }

    private static class IamError {
        private final String details;
        private final IamErrorType errorType;
        private final AccessServiceOuterClass.Subject subject;
        private final Throwable exception;

        public IamError(
            final IamErrorType errorType,
            final String details)
        {
            this(errorType, details, null);
        }

        public IamError(
            final IamErrorType errorType,
            final String details,
            final Throwable exception)
        {
            this(errorType, details, exception, null);
        }

        public IamError(
            final IamErrorType errorType,
            final String details,
            final Throwable exception,
            final AccessServiceOuterClass.Subject subject)
        {
            this.details = details;
            this.errorType = errorType;
            this.subject = subject;
            this.exception = exception;
        }

        public Throwable exception() {
            return exception;
        }

        public String details() {
            return details;
        }

        public IamErrorType errorType() {
            return errorType;
        }

        public AccessServiceOuterClass.Subject subject() {
            return subject;
        }

        @Override
        public String toString() {
            return "IamError{" +
                "details='" + details + '\'' +
                ", errorType=" + errorType +
                ", subject=" + subject +
                ", exception=" + exception +
                '}';
        }
    }
}
