package ru.yandex.search.yc.labels;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;

import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.yc.TokenRequiredYcSearchContext;
import ru.yandex.search.yc.YcSearchProxy;

class LabelsContext extends TokenRequiredYcSearchContext {
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);

    protected final ProxySession session;
    protected final JsonType jsonType;
    protected final AsyncClient client;
    protected final YcSearchProxy proxy;

    protected final boolean allowLaggingHosts;
    protected final boolean debug;
    protected final int length;
    protected final int offset;

    protected final Set<String> cloudIds;

    protected final Supplier<? extends HttpClientContext> contextGenerator;

    public LabelsContext(
        final YcSearchProxy proxy,
        final ProxySession session)
        throws BadRequestException {
        super(session);

        this.proxy = proxy;
        this.session = session;

        CgiParams params = session.params();

        List<String> cloudIds = params.getAll("cloud_id");
        if (cloudIds.isEmpty()) {
            this.cloudIds = params.get("cloud_ids", SET_PARSER);

            if (this.cloudIds.isEmpty()) {
                throw new BadRequestException("No cloud id supplied");
            }
        } else {
            this.cloudIds = new LinkedHashSet<>(cloudIds);
        }

        jsonType = JsonTypeExtractor.NORMAL.extract(params);
        allowLaggingHosts =
            params.getBoolean("allow-lagging-hosts", true);

        client = proxy.searchClient().adjust(session.context());

        debug = params.getBoolean("debug", false);
        length = params.get("length", 10, PositiveIntegerValidator.INSTANCE);
        offset = params.get("offset", 0, NonNegativeIntegerValidator.INSTANCE);

        contextGenerator =
            session.listener().createContextGeneratorFor(proxy.searchClient());
    }

    public ProxySession session() {
        return session;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public AsyncClient client() {
        return client;
    }

    public YcSearchProxy proxy() {
        return proxy;
    }

    public boolean allowLaggingHosts() {
        return allowLaggingHosts;
    }

    public boolean debug() {
        return debug;
    }

    public int length() {
        return length;
    }

    public int offset() {
        return offset;
    }

    public Set<String> cloudIds() {
        return cloudIds;
    }

    public Supplier<? extends HttpClientContext> contextGenerator() {
        return contextGenerator;
    }
}
