package ru.yandex.search.yc.labels;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.yc.BasicYcResultItem;
import ru.yandex.search.yc.YcConstants;
import ru.yandex.search.yc.YcSearchProxy;
import ru.yandex.search.yc.iam.AuthorizationResolution;
import ru.yandex.search.yc.iam.ResourceWithResolution;

public class LabelsHandler implements ProxyRequestHandler {
    private final YcSearchProxy proxy;
    private final static double LENGTH_COEFF = 1.25;

    public LabelsHandler(final YcSearchProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException {

        LabelsContext context = new LabelsContext(proxy, session);

        Map<SearchMapShard, List<User>> userMap = new LinkedHashMap<>();
        Map<SearchMapShard, Set<String>> cloudIdMap = new LinkedHashMap<>();
        for (String cloudId : context.cloudIds()) {
            User user = new User(YcConstants.YC_QUEUE, new StringPrefix(cloudId));
            SearchMapShard shard =
                proxy.searchMap().apply(user);
            userMap.computeIfAbsent(shard, (k) -> new ArrayList<>(context.cloudIds().size())).add(user);
            cloudIdMap.computeIfAbsent(shard, (k) -> new LinkedHashSet<>(context.cloudIds().size() << 1)).add(cloudId);
        }

        MultiFutureCallback<List<LabelsYcResultItem>> mfcb =
            new MultiFutureCallback<>(
                new ConcatCallback<>(
                    new LabelsResultPrinter(context),
                    context));

        for (Map.Entry<SearchMapShard, List<User>> entry: userMap.entrySet()) {
            LabelsShardContext labelsShardContext =
                new LabelsShardContext(proxy, context, entry, cloudIdMap.get(entry.getKey()));
            search(labelsShardContext,
                new SearchForCertainItemsCount(new TreeSet<>((label1, label2) ->
                        (label1.labelName().equals(label2.labelName()) &&
                         label1.labelValue().equals(label2.labelValue())) ? 0 : 1),
                     labelsShardContext,
                     mfcb.newCallback(),
                     0),
                0);
        }
        mfcb.done();
    }

    private static int lengthForSearchByContext(final LabelsShardContext context) {
        return (int) Math.ceil((context.offset() + context.length()) * LENGTH_COEFF);
    }

    private static void search(final LabelsShardContext context,
                               final FutureCallback<SearchResultWithItemsLeft<LabelsYcResultItem>> callback,
                               final int offset) throws HttpException {
        QueryConstructor query =
            new QueryConstructor(
                "/search-yc?IO_PRIO=0&json-type=dollar");

        query.append("service", YcConstants.YC_QUEUE);
        query.append("get", "*");
        query.append("length", lengthForSearchByContext(context));
        query.append("offset", offset);

        for (String cloudId: context.cloudIds()) {
            query.append("prefix", cloudId);
        }

        query.append("lowercase-expanded-terms", "true");
        query.append("replace-ee-expanded-terms", "true");

        query.append("text", "yc_atttype:label AND NOT yc_deleted_p:1");
        //query.append("dp", YcPlainSearchRule.LEFT_JOIN_DP);
        //query.append("group", "multi(yc_attname,yc_attvalue)");
        //query.append("merge_func", "none");

        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.proxy().config().searchFailOverDelay(),
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new SearchCallback(context, callback, offset));
    }

    private static final class SearchForCertainItemsCount
            implements FutureCallback<SearchResultWithItemsLeft<LabelsYcResultItem>> {
        private final Set<LabelsYcResultItem> previousResultItems;
        private final LabelsShardContext context;
        private final FutureCallback<? super List<LabelsYcResultItem>> callback;
        private final int offset;

        private SearchForCertainItemsCount(
                final Set<LabelsYcResultItem> previousResultItems,
                final LabelsShardContext context,
                final FutureCallback<? super List<LabelsYcResultItem>> callback,
                final int offset) {
            this.previousResultItems = previousResultItems;
            this.context = context;
            this.callback = callback;
            this.offset = offset;
        }

        @Override
        public void failed(final Exception ex) {
            callback.failed(ex);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        @Override
        public void completed(SearchResultWithItemsLeft<LabelsYcResultItem> result) {
            previousResultItems.addAll(result.resultItems);
            if ((previousResultItems.size() > (context.length() + context.offset())) || result.itemsLeft() == 0) {
                callback.completed(new ArrayList<>(previousResultItems));
            } else {
                int newOffset = this.offset + lengthForSearchByContext(context);
                try {
                    search(context, new SearchForCertainItemsCount(previousResultItems, context, callback, newOffset), newOffset);
                } catch (HttpException he) {
                    failed(he);
                }
            }
        }
    }

    private static final class SearchResultWithItemsLeft<T extends BasicYcResultItem>
    {
        private final List<T> resultItems;
        private final int itemsLeft;

        public SearchResultWithItemsLeft(final List<T> resultItems, final int itemsLeft) {
            this.resultItems = resultItems;
            this.itemsLeft = itemsLeft;
        }

        public List<T> resultItems() {
            return resultItems;
        }

        public int itemsLeft() {
            return itemsLeft;
        }
    }

    private static final class SearchCallback
        implements FutureCallback<JsonObject>
    {
        private final LabelsShardContext context;
        private final long fetchTime;
        private final FutureCallback<? super SearchResultWithItemsLeft<LabelsYcResultItem>> callback;
        private final int offset;

        private SearchCallback(
            final LabelsShardContext context,
            final FutureCallback<? super SearchResultWithItemsLeft<LabelsYcResultItem>> callback,
            final int offset)
        {
            this.context = context;
            this.fetchTime = System.currentTimeMillis();
            this.callback = callback;
            this.offset = offset;
        }

        @Override
        public void failed(final Exception ex) {
            callback.failed(ex);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        private void complete(
            final JsonMap map,
            final List<LabelsYcResultItem> result)
            throws JsonException
        {
            JsonList hits = map.getList("hitsArray");

            if (context.debug()) {
                context.logger().info(
                    "lucene response "
                        + JsonType.HUMAN_READABLE.toString(hits));
            }

            for (JsonObject jo: hits) {
                JsonMap item = jo.asMap();
                LabelsYcResultItem resultItem = new LabelsYcResultItem(item);
                if (!context.cloudIds().contains(resultItem.cloudId())) {
                    context.logger().warning(
                        "Wrong cloud id in response "
                            + JsonType.NORMAL.toString(resultItem)
                            + " expected " + context.cloudIds());
                    continue;
                }

                if (!context.proxy().config().allowedServices().test(
                    resultItem.service()))
                {
                    context.logger().warning(
                        "Service in response not in allowed list "
                            + JsonType.NORMAL.toString(resultItem)
                            + " allowed " + context.proxy().config().allowedServices());

                    continue;
                }

                result.add(resultItem);
            }
        }

        @Override
        public void completed(final JsonObject response) {
            context.logger().info(
                "Search finished in "
                    + (System.currentTimeMillis() - fetchTime) + "ms");

            List<LabelsYcResultItem> result = new ArrayList<>(lengthForSearchByContext(context));
            try {
                complete(response.asMap(), result);
                int hitsCount = response.asMap().getInt("hitsCount");

                MultiFutureCallback<ResourceWithResolution<LabelsYcResultItem>> mfcb =
                    new MultiFutureCallback<>(
                        new FilterCallback(
                            context.logger(),
                            result.size(),
                            hitsCount - result.size() - offset,
                            callback));

                context.logger().info("Result size, before filter " + result.size());

                for (LabelsYcResultItem item: result) {
                    context.proxy().resourceFilter().authorizePathes(
                        context.logger(),
                        context.iamToken(),
                        item,
                        mfcb.newCallback());
                }
                mfcb.done();
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private static class FilterCallback
        extends AbstractFilterFutureCallback<List<ResourceWithResolution<LabelsYcResultItem>>, SearchResultWithItemsLeft<LabelsYcResultItem>>
    {
        private final PrefixedLogger logger;
        private final int resources;
        private final int itemsLeft;

        public FilterCallback(
            final PrefixedLogger logger,
            final int resources,
            final int itemsLeft,
            final FutureCallback<? super SearchResultWithItemsLeft<LabelsYcResultItem>> callback)
        {
            super(callback);

            this.resources = resources;
            this.itemsLeft = itemsLeft;
            this.logger = logger;
        }

        @Override
        public void completed(
            final List<ResourceWithResolution<LabelsYcResultItem>> resourcesWithResolution)
        {
            int resolutionDrop = 0;
            List<LabelsYcResultItem> keepResources =
                new ArrayList<>(resourcesWithResolution.size());

            for (ResourceWithResolution<LabelsYcResultItem> res: resourcesWithResolution) {
                if (res.resolution() == AuthorizationResolution.ALLOW) {
                    keepResources.add(res.resource());
                } else {
                    resolutionDrop += 1;
                }
            }

            int totalDropped = resources - keepResources.size();
            if (totalDropped > 0) {
                StringBuilder log = new StringBuilder();
                log.append("Iam before filter: ");
                log.append(resources);
                log.append(" dropped by iam: ");
                log.append(resolutionDrop);
                log.append(" other drops: ");
                log.append(resources - resolutionDrop - keepResources.size());
                logger.warning(log.toString());
            }
            callback.completed(new SearchResultWithItemsLeft<>(keepResources, itemsLeft));
        }
    }

    private static class ConcatCallback<T extends BasicYcResultItem>
        extends AbstractFilterFutureCallback<List<List<T>>, List<T>>
    {
        private final LabelsContext context;

        public
        ConcatCallback(
            final FutureCallback<? super List<T>> callback,
            final LabelsContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final List<List<T>> results) {
            List<T> result =
                new ArrayList<T>(results.size() * context.length());
            for (List<T> item: results) {
                result.addAll(item);
            }

            callback.completed(result);
        }
    }
}
