package ru.yandex.search.yc.labels;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.logging.Level;

import com.google.protobuf.CodedOutputStream;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.search.yc.proto.LabelProto;

public class LabelsResultPrinter extends AbstractProxySessionCallback<List<LabelsYcResultItem>> {
    private final LabelsContext context;
    private static final ThreadLocal<ByteArrayOutputStream> OUT =
            ThreadLocal.withInitial(ByteArrayOutputStream::new);

    private final PrintFormat format;

    public LabelsResultPrinter(final LabelsContext context) throws BadRequestException {
        super(context.session());
        this.context = context;
        this.format = context.session().params().getEnum(PrintFormat.class, "format", PrintFormat.JSON);
    }

    @Override
    public void completed(final List<LabelsYcResultItem> result) {
        context.session().connection().setSessionInfo(
            SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
            Long.toString(result.size()));
        Set<LabelsYcResultItem> uniqueLabels =
            new TreeSet<>((label1, label2) ->
                (label1.labelName().equals(label2.labelName()) &&
                label1.labelValue().equals(label2.labelValue())) ? 0 : 1);
        boolean more = (context.offset + context.length) < result.size();
        uniqueLabels.addAll(result.subList(context.offset,
                Math.min(result.size(), context.offset + context.length)));
        switch (format) {
            case PROTO:
                printProto(uniqueLabels, more);
                break;
            case JSON:
            default:
                printJson(uniqueLabels, more);
                break;
        }
    }
    private static String noPrefix(String labelName) {
        return labelName.startsWith("label_") ? labelName.substring(6) : labelName;
    }
    private void printProto(final Set<LabelsYcResultItem> uniqueLabels, final boolean more) {
        LabelProto.TLabelResponse.Builder labelResponseBuilder = LabelProto.TLabelResponse.newBuilder();

        for (LabelsYcResultItem labelItem : uniqueLabels) {
            LabelProto.TLabel label = LabelProto.TLabel
                .newBuilder()
                .setLabelName(noPrefix(labelItem.labelName()))
                .setLabelValue(labelItem.labelValue())
                .build();
            labelResponseBuilder.addLabel(label);
        }
        labelResponseBuilder.setMore(more);
        ByteArrayOutputStream out = OUT.get();
        out.reset();
        CodedOutputStream codedOut = CodedOutputStream.newInstance(out);
        LabelProto.TLabelResponse labelResponse = labelResponseBuilder.build();
        try {
            labelResponse.writeTo(codedOut);
            codedOut.flush();
            byte[] postData = out.toByteArray();
            session.response(
                HttpStatus.SC_OK,
                new NByteArrayEntity(postData));
        } catch (IOException e) {
            session.logger().log(Level.SEVERE, "Protobuf response writing exception", e);
            failed(e);
        }
    }

    private void printJson(final Set<LabelsYcResultItem> uniqueLabels, final boolean more) {
        StringBuilderWriter sbw = new StringBuilderWriter();

        try (JsonWriter writer = context.jsonType().create(sbw)) {
            writer.startObject();
            writer.key("labels");
            writer.startArray();
            for (LabelsYcResultItem labelItem : uniqueLabels) {
                writer.startObject();
                writer.key(noPrefix(labelItem.labelName()));
                writer.value(labelItem.labelValue());
                writer.endObject();
            }
            writer.endArray();
            writer.key("more");
            writer.value(more);
            writer.endObject();
        } catch (IOException ioe) {
            failed(ioe);
            return;
        }

        session.response(
            HttpStatus.SC_OK,
            new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(context.session().acceptedCharset())));
    }

    private enum PrintFormat {
        JSON,
        PROTO;
    }
}
