package ru.yandex.search.yc.marketplace;

import java.io.IOException;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryParser;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.translit.LocaleEnTableSelector;
import ru.yandex.search.translit.LocaleRuTableSelector;
import ru.yandex.search.translit.Translit;
import ru.yandex.search.translit.TranslitTableSelector;
import ru.yandex.search.yc.MarketplaceFieldType;
import ru.yandex.search.yc.YcConstants;
import ru.yandex.search.yc.YcSearchProxy;

public class MarketPlaceSearchHandler implements ProxyRequestHandler {
    private static final int MINIMAL_TRANSLIT_LENGTH = 3;
    private static final Map<String, TranslitTableSelector> LANGUAGE_TRANSLITS;

    static {
        Map<String, TranslitTableSelector> table = new LinkedHashMap<>();
        table.put("ru", LocaleRuTableSelector.INSTANCE);
        table.put("en", LocaleEnTableSelector.INSTANCE);
        LANGUAGE_TRANSLITS = Collections.unmodifiableMap(table);
    }

    private final YcSearchProxy proxy;

    public MarketPlaceSearchHandler(final YcSearchProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        MarketplaceSearchContext context = new MarketplaceSearchContext(proxy, session);
        MarketplaceSearchPrinter callback = new MarketplaceSearchPrinter(context);
        QueryConstructor qc = new QueryConstructor("/search?marketplace");
        qc.append("prefix", context.prefix().toStringFast());
        qc.append("service", context.user().service());
        StringBuilder textSb = new StringBuilder();
        boolean luceneScorer = false;

        String language = context.locale().getLanguage();
        if (context.filter() != null) {
            MarketplaceLuceneQueryConstructor filterQuery =
                new MarketplaceLuceneQueryConstructor(context.project(), textSb);

            try {
                QueryAtom queryAtom = new QueryParser(context.filter()).parse();
                queryAtom.accept(filterQuery);
            } catch (ParseException pe) {
                callback.failed(pe);
                return;
            }

            if (filterQuery.language() != null && language.isEmpty()) {
                language = filterQuery.language();
            }

            if (context.sort().contains(MarketplaceFieldType.generateExtFieldName(context.project(), "categories_rank"))) {
                String category = filterQuery.categories().iterator().next();
                if (filterQuery.categories().size() <= 0) {
                    throw new BadRequestException("No category found in filter, but sorting by rank");
                }
                StringBuilder dpSb = new StringBuilder();
                dpSb.append("map_get(");
                dpSb.append(MarketplaceFieldType.generateIntFieldName(context.project(),   "categories_map"));
                dpSb.append(',');
                dpSb.append(category);
                dpSb.append(' ');
                dpSb.append(MarketplaceFieldType.generateExtFieldName(context.project(), "categories_rank"));
                dpSb.append(")");
                qc.append("dp", dpSb.toString());
            }
        }

        if (context.text() != null) {
            if (textSb.length() > 0) {
                textSb.append(" AND (");
            } else {
                textSb.append("(");
            }

            TranslitTableSelector selector = LANGUAGE_TRANSLITS.get(language);
            boolean translitEnabled = selector != null && context.text().length() >= MINIMAL_TRANSLIT_LENGTH;
            Set<String> translits;
            if (translitEnabled) {
                translits = Translit.process(context.text(), false, selector);
            } else {
                translits = Collections.emptySet();
            }

            if (translits.size() <= 0) {
                SearchRequestText.parseSuggest(
                    context.text(), context.locale())
                    .fieldsQuery(textSb, context.searchFields());
            } else {
                luceneScorer = true;
                int prev = textSb.length();
                textSb.append('(');
                SearchRequestText.parseSuggest(
                    context.text(), context.locale())
                    .fieldsQuery(textSb, context.searchFields());
                textSb.append(')');
                if (textSb.length() - 2 == prev) {
                    textSb.setLength(textSb.length() - 2);
                } else {
                    textSb.append("^10.0 OR ");
                }
                for (String request: translits) {
                    prev = textSb.length();
                    textSb.append('(');
                    SearchRequestText
                        .parseSuggest(request, context.locale())
                        .fieldsQuery(textSb, context.searchFields());
                    textSb.append(')');
                    if (textSb.length() - 2 == prev) {
                        textSb.setLength(textSb.length() - 2);
                    } else {
                        textSb.append("^1.0");
                    }

                    textSb.append(" OR ");
                }
                textSb.setLength(textSb.length() - 4);
            }

            textSb.append(")");
        }

        qc.append("text", textSb.toString());
        qc.append("length", context.offset() + context.length());
        qc.append("get", "id," + MarketplaceFieldType.generateIntFieldName(context.project(), "data"));
        if (!(context.sort().size() == 1 && context.defaultSort.equalsIgnoreCase(context.sort().iterator().next()))) {
            qc.append("sort", String.join(",", context.sort()));
        } else if (luceneScorer) {
            qc.append("scorer", "lucene");
            qc.append("sort", "#score");
        }

        if (!context.sort().isEmpty() && context.sortOrder() != SortOrder.DESC) {
            qc.append("asc", "true");
        }

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            context.failoverDelay(),
            context.localityShuffle(),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            callback);
    }

    private enum SortOrder {
        ASC,
        DESC
    }

    private static class MarketplaceSearchContext
        implements UniversalSearchProxyRequestContext
    {
        protected final ProxySession session;
        protected final Supplier<? extends HttpClientContext> contextGenerator;
        protected final YcSearchProxy proxy;
        protected final User user;

        protected final Prefix prefix;
        protected final PrefixedLogger logger;

        private final int offset;
        private final int length;
        private final Set<String> sort;
        private final String filter;
        private final String text;
        private final String project;
        private final String defaultSort;
        private final SortOrder sortOrder;
        private final Set<String> searchFields;
        private final Set<String> getFields;
        private final AsyncClient client;
        private final Locale locale;
        private final JsonType jsonType;
        private final boolean localityShuffle;
        private final long failoverDelay;
        private final MarketplaceFieldParser fieldParser;

        public MarketplaceSearchContext(
            final YcSearchProxy proxy,
            final ProxySession session)
            throws BadRequestException
        {
            this.proxy = proxy;
            this.session = session;

            CgiParams params = session.params();

            jsonType = JsonTypeExtractor.NORMAL.extract(params);
            this.logger = session.logger();

            client = proxy.searchClient().adjust(session.context());
            contextGenerator =
                session.listener().createContextGeneratorFor(client);

            getFields = params.get(
                "get",
                new LinkedHashSet<>(Arrays.asList("id", "data")),
                new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new));

            offset = params.getInt("offset", 0);
            length = params.getInt("length", 10);
            project = params.getString("project", "default");

            prefix = new StringPrefix(MarketplaceFieldType.prefix(project));
            user = new User(YcConstants.YC_MARKETPLACE_QUEUE, prefix);
            defaultSort = MarketplaceFieldType.generateExtFieldName(this.project, "default");
            fieldParser = new MarketplaceFieldParser(this.project);
            sortOrder = params.getEnum(SortOrder.class, "sort_order", SortOrder.DESC);
            localityShuffle = params.getBoolean("locality-shuffle", true);
            failoverDelay = params.getLong("failover-delay", 300L);
            locale = params.getLocale("locale", Locale.ROOT);
            filter = params.getString("filter", null);
            text = params.getString("text", null);
            if (filter == null && text == null) {
                throw new BadRequestException("text or filter param should be supplied");
            }
            sort =
                params.get(
                    "sort",
                    Collections.singleton(defaultSort),
                    new CollectionParser<>(
                        fieldParser,
                        LinkedHashSet::new));
            searchFields =
                params.get(
                    "search_fields",
                    Collections.emptySet(),
                    new CollectionParser<>(
                        fieldParser,
                        LinkedHashSet::new));
            if (text != null && searchFields.isEmpty()) {
                throw new BadRequestException("Text parameter set, but no search fields supplied");
            }
        }

        public String project() {
            return project;
        }

        public Prefix prefix() {
            return prefix;
        }

        public SortOrder sortOrder() {
            return sortOrder;
        }

        public Set<String> getFields() {
            return getFields;
        }

        public boolean localityShuffle() {
            return localityShuffle;
        }

        public long failoverDelay() {
            return failoverDelay;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return logger;
        }

        @Override
        public long lagTolerance() {
            return Long.MAX_VALUE;
        }

        public int offset() {
            return offset;
        }

        public int length() {
            return length;
        }

        public Set<String> sort() {
            return sort;
        }

        public Locale locale() {
            return locale;
        }

        public String filter() {
            return filter;
        }

        public String text() {
            return text;
        }

        public Set<String> searchFields() {
            return searchFields;
        }

        public Supplier<? extends HttpClientContext> contextGenerator() {
            return contextGenerator;
        }
    }

    private static class MarketplaceSearchPrinter
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final MarketplaceSearchContext context;
        public MarketplaceSearchPrinter(final MarketplaceSearchContext context) {
            super(context.session());

            this.context = context;
        }

        private void writeDoc(
            final String project,
            final JsonWriter writer,
            final JsonMap doc)
            throws IOException, JsonException
        {
            JsonObject data =
                TypesafeValueContentHandler.parse(doc.getString(MarketplaceFieldType.generateIntFieldName(project, "data"))).asMap();
            writer.startObject();
            writer.key("id");
            writer.value(data.get("id"));
            if (context.getFields().contains("data")) {
                writer.key("data");
                writer.value(data);
            }

            writer.endObject();
        }

        @Override
        public void completed(final JsonObject searchResultObj) {
            StringBuilderWriter sbw = new StringBuilderWriter();

            try (JsonWriter writer = context.jsonType().create(sbw)) {
                JsonMap searchResult = searchResultObj.asMap();
                int hitsCount = searchResult.getInt("hitsCount");
                JsonList hitsArray = searchResult.getList("hitsArray");
                boolean hasNext = hitsArray.size() >= context.length() + context.offset;
                context.session().connection().setSessionInfo(
                    SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
                    Long.toString(hitsArray.size()));

                writer.startObject();
                writer.key("has_next");
                writer.value(hasNext);
                writer.key("total");
                writer.value(hitsCount);
                writer.key("documents");
                writer.startArray();
                for (int i = context.offset; i < Math.min(hitsArray.size(), context.offset + context.length); i++) {
                    writeDoc(context.project(), writer, hitsArray.get(i).asMap());
                }
                writer.endArray();
                writer.endObject();
            } catch (JsonException | IOException e) {
                failed(e);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
        }
    }
}
