package ru.yandex.search.yc.marketplace;

import java.io.IOException;
import java.text.ParseException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryParser;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.translit.LocaleEnTableSelector;
import ru.yandex.search.translit.LocaleRuTableSelector;
import ru.yandex.search.translit.Translit;
import ru.yandex.search.translit.TranslitTableSelector;
import ru.yandex.search.yc.MarketplaceFieldType;
import ru.yandex.search.yc.YcConstants;
import ru.yandex.search.yc.YcSearchProxy;

public class MarketplaceAggregationHandler implements ProxyRequestHandler {
    private static final int MINIMAL_TRANSLIT_LENGTH = 3;

    private static final Map<String, TranslitTableSelector> LANGUAGE_TRANSLITS;

    static {
        Map<String, TranslitTableSelector> table = new LinkedHashMap<>();
        table.put("ru", LocaleRuTableSelector.INSTANCE);
        table.put("en", LocaleEnTableSelector.INSTANCE);
        LANGUAGE_TRANSLITS = Collections.unmodifiableMap(table);
    }

    private final YcSearchProxy proxy;

    public MarketplaceAggregationHandler(final YcSearchProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        MarketplaceAggregationContext context = new MarketplaceAggregationContext(proxy, session);
        MarketplaceAggragationPrinter callback = new MarketplaceAggragationPrinter(context);
        QueryConstructor qc = new QueryConstructor("/printkeys?marketplace");
        qc.append("prefix", context.prefix().toStringFast());
        qc.append("user", context.prefix().toStringFast());
        qc.append("service", context.user().service());
        qc.append("field", context.field());
        StringBuilder textSb = new StringBuilder();

        String language = context.locale().getLanguage();
        if (context.filter() != null) {
            if (textSb.length() > 0) {
                textSb.append(" AND (");
            } else {
                textSb.append("(");
            }

            MarketplaceLuceneQueryConstructor filterQuery =
                new MarketplaceLuceneQueryConstructor(context.project, textSb);

            try {
                QueryAtom queryAtom = new QueryParser(context.filter()).parse();
                queryAtom.accept(filterQuery);
            } catch (ParseException pe) {
                callback.failed(pe);
                return;
            }

            if (filterQuery.language() != null && language.isEmpty()) {
                language = filterQuery.language();
            }

            textSb.append(")");
        }

        if (context.text() != null) {
            if (textSb.length() > 0) {
                textSb.append(" AND (");
            } else {
                textSb.append("(");
            }

            TranslitTableSelector selector = LANGUAGE_TRANSLITS.get(language);
            boolean translitEnabled = selector != null && context.text().length() >= MINIMAL_TRANSLIT_LENGTH;
            Set<String> translits;
            if (translitEnabled) {
                translits = Translit.process(context.text(), false, selector);
            } else {
                translits = Collections.emptySet();
            }

            if (translits.size() <= 0) {
                SearchRequestText.parseSuggest(
                    context.text(), context.locale())
                    .fieldsQuery(textSb, context.searchFields());
            } else {
                SearchRequestText.parseSuggest(
                    context.text(), context.locale())
                    .fieldsQuery(textSb, context.searchFields());
                textSb.append(" OR ");
                for (String request: translits) {
                    SearchRequestText
                        .parseSuggest(request, context.locale())
                        .fieldsQuery(textSb, context.searchFields());
                    textSb.append(" OR ");
                }
                textSb.setLength(textSb.length() - 4);
            }

            textSb.append(")");
        }

        if (textSb.length() > 0) {
            qc.append("text", textSb.toString());
        }

        qc.append("print-freqs", "true");
        qc.append("max-freq", "0");
        qc.append("skip-deleted", "true");
        qc.append("json-type", "normal");

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            context.failoverDelay(),
            context.localityShuffle(),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            callback);
    }

    private static class MarketplaceAggregationContext
        implements UniversalSearchProxyRequestContext
    {
        protected final ProxySession session;
        protected final Supplier<? extends HttpClientContext> contextGenerator;
        protected final YcSearchProxy proxy;
        protected final User user;

        protected final Prefix prefix;
        protected final PrefixedLogger logger;

        private final int offset;
        private final int length;
        private final String text;
        private final Set<String> searchFields;
        private final String filter;
        private final String field;
        private final String project;
        private final AsyncClient client;
        private final JsonType jsonType;
        private final boolean localityShuffle;
        private final long failoverDelay;
        private final Locale locale;
        private final MarketplaceFieldParser fieldParser;

        public MarketplaceAggregationContext(
            final YcSearchProxy proxy,
            final ProxySession session)
            throws BadRequestException
        {
            this.proxy = proxy;
            this.session = session;

            CgiParams params = session.params();

            jsonType = JsonTypeExtractor.NORMAL.extract(params);
            this.logger = session.logger();

            client = proxy.searchClient().adjust(session.context());
            contextGenerator =
                session.listener().createContextGeneratorFor(client);

            offset = params.getInt("offset", 0);
            length = params.getInt("length", 10);
            project = params.getString("project", "default");
            fieldParser = new MarketplaceFieldParser(project);

            prefix = new StringPrefix(MarketplaceFieldType.prefix(project));
            user = new User(YcConstants.YC_MARKETPLACE_QUEUE, prefix);

            localityShuffle = params.getBoolean("locality-shuffle", true);
            failoverDelay = params.getLong("failover-delay", 300L);

            filter = params.getString("filter", null);
            field = params.get("field", fieldParser);
            text = params.getString("text", null);
            searchFields =
                params.get(
                    "search_fields",
                    Collections.emptySet(),
                    new CollectionParser<>(
                        fieldParser,
                        LinkedHashSet::new));
            locale = params.getLocale("locale", Locale.ROOT);
        }

        public Prefix prefix() {
            return prefix;
        }

        public String project() {
            return project;
        }

        public String text() {
            return text;
        }

        public Set<String> searchFields() {
            return searchFields;
        }

        public Locale locale() {
            return locale;
        }

        public boolean localityShuffle() {
            return localityShuffle;
        }

        public long failoverDelay() {
            return failoverDelay;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public String field() {
            return field;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return logger;
        }

        @Override
        public long lagTolerance() {
            return Long.MAX_VALUE;
        }

        public int offset() {
            return offset;
        }

        public int length() {
            return length;
        }

        public String filter() {
            return filter;
        }

        public Supplier<? extends HttpClientContext> contextGenerator() {
            return contextGenerator;
        }
    }

    private static class MarketplaceAggragationPrinter
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final MarketplaceAggregationContext context;
        public MarketplaceAggragationPrinter(final MarketplaceAggregationContext context) {
            super(context.session());

            this.context = context;
        }

        @Override
        public void completed(final JsonObject searchResultObj) {
            StringBuilderWriter sbw = new StringBuilderWriter();

            System.out.println(JsonType.HUMAN_READABLE.toString(searchResultObj));
            try (JsonWriter writer = context.jsonType().create(sbw)) {
                JsonMap searchResult = searchResultObj.asMap();

                writer.startObject();
                writer.key("data");
                writer.startObject();
                for (Map.Entry<String, JsonObject> entry: searchResult.entrySet()) {
                    String key = entry.getKey();
                    int index = key.indexOf('#');
                    if (index < 0 || index >= key.length() - 1) {
                        context.logger().warning(
                            "Invalid record in printkeys " + JsonType.NORMAL.toString(searchResultObj));
                        continue;
                    }
                    writer.key(key.substring(index + 1));
                    writer.value(entry.getValue().asMap().getInt("freq"));
                }
                writer.endObject();
                writer.endObject();
            } catch (JsonException | IOException e) {
                failed(e);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
        }
    }
}
