package ru.yandex.search.yc.marketplace;

import java.text.ParseException;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import ru.yandex.parser.query.AndQuery;
import ru.yandex.parser.query.FieldQuery;
import ru.yandex.parser.query.NotQuery;
import ru.yandex.parser.query.OrQuery;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryToken;
import ru.yandex.parser.query.QueryVisitor;
import ru.yandex.parser.query.QuotedQuery;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.yc.MarketplaceFieldType;

public class MarketplaceLuceneQueryConstructor
    implements QueryVisitor<Void, ParseException>
{
    public static final String OR = " OR ";

    protected final StringBuilder request;
    protected final Set<String> categories;
    protected final String project;
    protected String language = null;


    protected MarketplaceLuceneQueryConstructor(
        final String project,
        final StringBuilder textSb)
    {
        this.project = project;
        request = textSb;
        categories = new LinkedHashSet<>();
    }

    @Override
    public Void visit(final AndQuery query) throws ParseException {
        request.append('(');
        visit(query.lhs());
        request.append(" AND ");
        // optimize a bit: skip NotQuery visit and selectAll request usage
        QueryAtom rhs = query.rhs();
        if (rhs instanceof NotQuery) {
            request.append("NOT ");
            visit(((NotQuery) rhs).query());
        } else {
            visit(rhs);
        }
        request.append(')');
        return null;
    }

    @Override
    public Void visit(final OrQuery query) throws ParseException {
        request.append('(');
        visit(query.lhs());
        request.append(OR);
        visit(query.rhs());
        request.append(')');
        return null;
    }

    @Override
    public Void visit(final NotQuery query) throws ParseException {
        request.append('(');
        request.append("__prefix:");
        request.append(MarketplaceFieldType.prefix(project));
        request.append(" AND NOT ");
        visit(query.query());
        request.append(')');
        return null;
    }

    @Override
    public Void visit(final FieldQuery query) throws ParseException {
        List<String> fields = query.fields();
        QueryAtom subquery = query.query();

        String field = fields.get(0).replaceAll("\\.", "_");
        if ("categories_id".equalsIgnoreCase(field) && subquery instanceof QueryToken) {
            categories.add(((QueryToken) subquery).text().trim());
        }

        if ("language".equalsIgnoreCase(field) && subquery instanceof QueryToken) {
            language = ((QueryToken) subquery).text().trim();
        }

        request.append(MarketplaceFieldType.generateExtFieldName(project, field));
        request.append(':');

        visit(subquery);
        return null;
    }

    public String language() {
        return language;
    }

    @Override
    public Void visit(final QueryToken query) throws ParseException {
        request.append(SearchRequestText.fullEscape(query.text(), true));
        return null;
    }

    @Override
    public Void visit(final QuotedQuery query) throws ParseException {
        request.append('(');
        boolean first = true;
        for (QueryToken token: query.tokens()) {
            if (first) {
                first = false;
            } else {
                request.append(' ');
            }
            request.append(
                token.text().replace('"', ' ').replace('\\', ' '));
        }
        request.append(')');
        return null;
    }

    public Set<String> categories() {
        return categories;
    }
}
