#pragma once

#include <macs/types.h>
#include <boost/optional.hpp>
#include <mail/sendbernar/client/include/exception.h>


namespace sendbernar::params {

using Stid = std::string;
using Mid = macs::Mid;
using Mailbox = std::string;
using Name = std::string;
using Subject = std::string;
using Text = std::string;
using Lids = std::vector<std::string>;
using Mids = std::vector<Mid>;
using RecipientsStr = std::string;
using References = std::string;
using CaptchaValue = std::string;
using CaptchaType = std::string;
using CaptchaKey = std::string;
using Header = std::string;
using Uid = macs::Uid;
using MessageId = std::string;
using Mention = std::string;
using Mentions = std::vector<Mention>;

template <typename T>
using opt = boost::optional<T>;

struct Sender {
    opt<Mailbox> from_mailbox;
    opt<Name> from_name;
};

struct Message {
    opt<Subject> subj;
    opt<Text> text;
    opt<bool> html;
    opt<Mid> source_mid;
    opt<MessageId> message_id;
    opt<std::time_t> current_time;
    opt<bool> force7bit;
};

struct Recipients {
    opt<RecipientsStr> to;
    opt<RecipientsStr> cc;
    opt<RecipientsStr> bcc;
};

struct Delivery {
    opt<std::time_t> noanswer_remind_period;
    opt<bool> confirm_delivery;
};

struct Captcha {
    CaptchaValue captcha_entered;
    CaptchaKey captcha_key;
};

struct NoAnswerReminder {
    Header message_id;
    unsigned no_answer_period;
};

struct IdetifiableMessagePart {
    Mid mid;
    std::string hid;

    IdetifiableMessagePart() {}
    IdetifiableMessagePart(const std::string& mid, const std::string& hid) : mid(mid), hid(hid) { }
    bool operator==(const IdetifiableMessagePart&) const = default;
};

enum class MarkAs {
    replied,
    forwarded
};

inline MarkAs markAsFromString(const std::string& str) {
    if (str == "forwarded") {
        return MarkAs::forwarded;
    } else if (str == "replied") {
        return MarkAs::replied;
    } else {
        throw Exception("cannot convert str to MarkAs: "+str);
    }
}

inline std::string markAsToString(const MarkAs& m) {
    switch(m) {
        case MarkAs::forwarded:
            return "forwarded";
        case MarkAs::replied:
            return "replied";
        default:
            throw Exception("strange MarkAs: " + std::to_string(static_cast<int>(m)));
    }
}

struct InReplyTo {
    std::string inreplyto;
    MarkAs mark_as = MarkAs::replied;
};

using Parts = std::vector<IdetifiableMessagePart>;

struct DiskAttach {
    std::string path;
    std::string name;
    opt<std::string> hash;
    opt<std::string> preview_path;
    opt<std::size_t> size;
    opt<std::string> folder;
    bool operator==(const DiskAttach&) const = default;
};

struct Attaches {
    opt<std::vector<Stid>> uploaded_attach_stids;
    opt<std::string> disk_attaches;
    opt<std::vector<DiskAttach>> disk_attaches_json;
    opt<Mids> forward_mids;
    opt<Parts> parts_json;
};

/*===========================================================================
 * Send methods parameters
 *===========================================================================*/

struct SaveTemplate {
    Attaches attaches;
    Sender sender;
    Message message;
    Recipients recipients;
    opt<Lids> lids;
};

struct SaveDraft {
    Attaches attaches;
    Sender sender;
    Message message;
    Recipients recipients;
    opt<References> references;
    opt<std::string> inreplyto;
    opt<Lids> lids;
};

struct SendMessage {
    Attaches attaches;
    Sender sender;
    Message message;
    Delivery delivery;
    Recipients recipients;
    opt<Lids> lids;
    opt<MarkAs> markAs;
    opt<Captcha> captcha;
    opt<InReplyTo> inreplyto;
    opt<References> references;
    opt<CaptchaType> captcha_type;
    opt<Mentions> mentions;
    opt<std::string> operation_id;
    opt<bool> captcha_passed;
};

struct SendShare {
    Attaches attaches;
    Message message;
    Recipients recipients;
    opt<Lids> lids;
    opt<std::string> inreplyto;
    opt<References> references;
    std::string admin_uid;
};

struct SendDelayed {
    SendMessage send;
    std::time_t send_time;
    bool relative = false;
};

struct CancelSendDelayed {
    Mid mid;
};

struct ListUnsubscribe {
    std::string to;
    opt<std::string> subject;
    opt<std::string> body;
    opt<Mailbox> from_mailbox;
};

struct LimitsParams {
    Recipients recipients;
};

enum class BarbetMessageType {
    create_failed,
    restore_failed
};

inline BarbetMessageType barbetMessageTypeFromString(const std::string& str) {
    if (str == "create_failed") {
        return BarbetMessageType::create_failed;
    } else if (str == "restore_failed") {
        return BarbetMessageType::restore_failed;
    } else {
        throw Exception("cannot convert str to BarbetMessageType: "+str);
    }
}

inline std::string barbetMessageTypeToString(const BarbetMessageType& m) {
    switch(m) {
        case BarbetMessageType::create_failed:
            return "create_failed";
        case BarbetMessageType::restore_failed:
            return "restore_failed";
        default:
            throw Exception("strange BarbetMessageType: " + std::to_string(static_cast<int>(m)));
    }
}

struct SendBarbetMessage {
    std::string uniq_id;
    BarbetMessageType type = BarbetMessageType::create_failed;
};

/*===========================================================================
 * Compose methods parameters
 *===========================================================================*/

struct ComposeMessage {
    Attaches attaches;
    Sender sender;
    Message message;
    Recipients recipients;
    opt<References> references;
    opt<std::string> inreplyto;
    opt<Lids> lids;
};

struct ComposeDraft {
    Attaches attaches;
    Sender sender;
    Message message;
    Recipients recipients;
    opt<References> references;
    opt<std::string> inreplyto;
    opt<Lids> lids;
};

struct PostProcessMessage {
    std::string nonempty_subject;
    std::string message_id;
    std::time_t message_date = 0;
    Recipients nonempty_recipients;
    opt<Lids> lids;
    opt<Mid> source_mid;
    opt<std::time_t> noanswer_remind_period;
    opt<InReplyTo> inreplyto;
    opt<Mids> forward_mids;
};

/*===========================================================================
 * Callbacks parameters
 *===========================================================================*/

struct ContinueSendingMessage {
    Mid mid;
    bool notify_on_send;
    opt<Mentions> mentions;
    PostProcessMessage postprocess;
};

struct RemindMessageCallback {
    Mid mid;
    std::string lang;
    std::string date;
    std::string account;
};

struct NoAnswerRemindCallback {
    std::string message_id;
    std::string lang;
    std::string date;
};


/*===========================================================================
 * WriteAttach parameters
 *===========================================================================*/

struct WriteAttachParams {
    std::string filename;
    std::string raw_body;
};


/*===========================================================================
 * Common parameters
 *===========================================================================*/

struct CommonParams {
    std::string caller;
    Uid uid;
    std::string requestId;
    std::string realIp;
    std::string originalHost;
};

struct UserJournalParams {
    std::string connectionId;
    std::string expBoxes;
    std::string enabledExpBoxes;
    std::string clientType;
    std::string clientVersion;
    std::string userAgent;
    std::string yandexUid;
    std::string iCookie;
};

template<class Params>
inline const auto& normalize(const Params& params) {
    if constexpr (std::is_same<Params, params::SendDelayed>::value) {
        return params.send;
    } else {
        return params;
    }
}

}
