'use strict';

const package_version = require('./package.json').version;

const in_reply_to = {
    inreplyto: {
        name: 'inreplyto',
        required: false
    },
    mark_as: {
        name: 'mark_as',
        required: false
    }
};

const attaches = {
    uploaded_attach_stids: {
        name: 'uploaded_attach_stids',
        required: false
    },
    disk_attaches: {
        name: 'disk_attaches',
        required: false
    },
    disk_attaches_json: {
        name: 'disk_attaches_json',
        required: false
    },
    forward_mids: {
        name: 'forward_mids',
        required: false
    },
    parts_json: {
        name: 'parts_json',
        required: false
    }
};

const sender = {
    from_mailbox: {
        name: 'from_mailbox',
        required: false
    },
    from_name: {
        name: 'from_name',
        required: false
    }
};

const message = {
    subj: {
        name: 'subj',
        required: false
    },
    text: {
        name: 'text',
        required: false
    },
    html: {
        name: 'html',
        required: false
    },
    force7bit: {
        name: 'force7bit',
        required: false
    },
    source_mid: {
        name: 'source_mid',
        required: false
    },
    message_id: {
        name: 'message_id',
        required: false
    }, current_time: {
        name: 'current_time',
        required: false
    }
};

const recipients = {
    to: {
        name: 'to',
        required: false
    },
    cc: {
        name: 'cc',
        required: false
    },
    bcc: {
        name: 'bcc',
        required: false
    }
};

const delivery = {
    noanswer_remind_period: {
        name: 'noanswer_remind_period',
        required: false
    },
    confirm_delivery: {
        name: 'confirm_delivery',
        required: false
    }
};

const captcha_legacy = {
    captcha_entered: {
        name: 'captcha_entered',
        required: false
    },
    captcha_key: {
        name: 'captcha_key',
        required: false
    },
    captcha_type: {
        name: 'captcha_type',
        required: false
    }
};

const captcha = {
    captcha_passed: {
        name: 'captcha_passed',
        required: false
    }
};

const operation = {
    operation_id: {
        name: 'operation_id',
        required: false
    }
};

const lids = {
    lids: {
        name: 'lids',
        required: false
    }
};

const references = {
    references: {
        name: 'references',
        required: false
    }
};

const save_template_schema = Object.assign({}, attaches, sender, message, recipients, lids);

const save_draft_schema = Object.assign({}, attaches, sender, message, recipients, references, lids, {
    inreplyto: {
        name: 'inreplyto',
        required: false
    }
});

const send_message_schema = Object.assign({}, attaches, sender, message, delivery, recipients,
    captcha_legacy, captcha, lids, references, in_reply_to, operation, {
        mentions: {
            name: 'mentions',
            required: false
        },
    });

const send_delayed_schema = Object.assign({}, send_message_schema, {
    send_time: {
        name: 'send_time',
        required: true
    }
});

const send_service_schema = send_message_schema;

const send_share_schema = Object.assign({}, attaches, message, recipients, lids, references, {
        inreplyto: {
            name: 'inreplyto',
            required: true
        }, admin_uid: {
            name: 'admin_uid',
            required: true
        }
    }
);

const cancel_send_delayed_schema = {
    mid: {
        name: 'mid',
        required: true
    }
};

const list_unsubscribe_schema = {
    to: {
        name: 'to',
        required: true
    },
    subject: {
        name: 'subject',
        required: false
    },
    body: {
        name: 'body',
        required: false
    },
    from_mailbox: {
        name: 'from_mailbox',
        required: false
    }
};

const transform_object = (obj) => {
    if (Array.isArray(obj)) {
        const ret = [];

        obj.forEach((el) => {
            ret.push(transform_object(el));
        });

        return ret;
    } else if (typeof obj === 'boolean') {
        return obj ? 'yes' : 'no';
    } else if (typeof obj === 'number') {
        return obj.toString();
    } else if (typeof obj === 'object') {
        return JSON.stringify(obj);
    }

    return obj;
};

const fill_request = (request, section, reflection, object) => {
    for (const key in reflection) {
        const obj = reflection[key];

        const empty = !Object.prototype.hasOwnProperty.call(object, key) || object[key] === undefined;

        if (!empty) {
            request[section][obj.name] = transform_object(object[key]);
        } else if (obj.required && empty) {
            throw Error('missing required property: ' + key);
        }
    }

    return request;
};

const common_params = (params, request) => {
    const as_get = {
        caller: {
            name: 'caller',
            required: false
        },
        uid: {
            name: 'uid',
            required: true
        }
    };

    const as_headers = {
        requestId: {
            name: 'X-Request-Id',
            required: false
        },
        realIp: {
            name: 'X-Real-Ip',
            required: false
        },
        originalHost: {
            name: 'X-Original-Host',
            required: false
        }
    };

    fill_request(request, 'query', as_get, params);
    fill_request(request, 'headers', as_headers, params);

    return request;
};

const user_journal = (params, request) => {
    const as_headers = {
        connectionId: {
            name: 'connection_id',
            required: true
        },
        expBoxes: {
            name: 'X-Yandex-ExpBoxes',
            required: true
        },
        enabledExpBoxes: {
            name: 'X-Yandex-EnabledExpBoxes',
            required: true
        },
        clientType: {
            name: 'X-Yandex-ClientType',
            required: true
        },
        clientVersion: {
            name: 'X-Yandex-ClientVersion',
            required: true
        },
        yandexUid: {
            name: 'yandexuid',
            required: true
        },
        iCookie: {
            name: 'icookie',
            required: false
        },
        userAgent: {
            name: 'User-Agent',
            required: true
        }
    };

    return fill_request(request, 'headers', as_headers, params);
};

const flatten = (object, ret) => {
    for (const key in object) {
        // eslint-disable-next-line eqeqeq
        if (object[key] == null) {
            continue;
        }

        const obj = object[key].valueOf();

        if (typeof obj === 'object') {
            if (Array.isArray(obj)) {
                ret[key] = obj;
            } else {
                flatten(obj, ret);
            }
        } else {
            ret[key] = obj;
        }
    }

    return ret;
};

const make_clean_request = () => {
    return {
        path: '',
        query: {},
        body: {},
        headers: {}
    };
};

const make_clean_post_body_request = () => {
    return {
        path: '',
        query: {},
        body: '',
        headers: {}
    };
};

const make_params = (url, common, uj, params, schema) => {
    const request = make_clean_request();
    request.path = url;
    common_params(common, request);
    user_journal(uj, request);
    params = flatten(params, {});
    fill_request(request, 'body', schema, params);

    for (const key in params) {
        if (!Object.prototype.hasOwnProperty.call(schema, key)) {
            throw Error('there is strange property: ' + key + ', ' + JSON.stringify(schema));
        }
    }

    return request;
};

const make_post_params = (url, common, uj, params) => {
    const request = make_clean_post_body_request();
    request.path = url;
    common_params(common, request);
    user_journal(uj, request);

    request.query.filename = params.filename;
    request.body = params.body;

    return request;
};

const add_sendbernar_client_info = (request) => {
    request.query.v = '1';
    request.query.sc = 'js_' + package_version;

    return request;
};

const save_draft = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/save_draft', common, uj, params, save_draft_schema)
    );
};
const save_template = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/save_template', common, uj, params, save_template_schema)
    );
};
const send_message = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/send_message', common, uj, params, send_message_schema)
    );
};
const send_service = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/send_service', common, uj, params, send_service_schema)
    );
};
const send_delayed = (common, uj, params) => {
    const request = make_params('/send_delayed', common, uj, params, send_delayed_schema);
    if (request.body.send_time < 0) {
        throw Error('send_time cannot be less than zero');
    }
    return add_sendbernar_client_info(request);
};
const send_share = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/send_share', common, uj, params, send_share_schema)
    );
};
const send_undo = (common, uj, params) => {
    const request = make_params('/send_undo', common, uj, params, send_delayed_schema);
    if (request.body.send_time < 0) {
        throw Error('send_time cannot be less than zero');
    }
    return add_sendbernar_client_info(request);
};
const cancel_send_delayed = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/cancel_send_delayed', common, uj, params, cancel_send_delayed_schema)
    );
};
const cancel_send_undo = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/cancel_send_undo', common, uj, params, cancel_send_delayed_schema)
    );
};
const list_unsubscribe = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/list_unsubscribe', common, uj, params, list_unsubscribe_schema)
    );
};
const write_attachment = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_post_params('/write_attachment', common, uj, params)
    );
};
const limits = (common, uj, params) => {
    return add_sendbernar_client_info(
        make_params('/limits', common, uj, params, recipients)
    );
};
const generate_operation_id = (common, uj) => {
    return add_sendbernar_client_info(
        make_params('/generate_operation_id', common, uj, {})
    );
};

module.exports = {
    fill_request,
    common_params,
    user_journal,
    flatten,
    transform_object,
    make_clean_request,
    save_draft,
    save_template,
    send_message,
    send_service,
    send_delayed,
    send_share,
    send_undo,
    cancel_send_delayed,
    cancel_send_undo,
    list_unsubscribe,
    write_attachment,
    limits,
    generate_operation_id
};
