'use strict';

const is_internal_error = (status) => {
    // eslint-disable-next-line no-bitwise
    return (status / 100 | 0) === 5 && status !== 505;
};

const handle_bad_request_reason = (resp) => {
    if (resp !== undefined) {
        const r = resp.reason;
        const significant = r === 'incorrect_to' ||
                            r === 'incorrect_cc' ||
                            r === 'incorrect_bcc' ||
                            r === 'no_recipients' ||
                            r === 'max_email_addr_reached' ||
                            r === 'attachment_too_big';
        return significant ? r : undefined;
    }
    return undefined;
};

const handle_ban_reason = (resp) => {
    if (resp !== undefined) {
        return resp.banReason;
    }
    return undefined;
};

const handle_spam_reason = (resp) => {
    if (resp !== undefined) {
        const r = resp.reason;
        const significant = r === 'virus_found' ||
                            r === 'strongspam_found';
        return significant ? r : undefined;
    }
    return undefined;
};

const common_codes = (status, resp) => {
    if (status === 200) {
        return { status: 'ok' };
    } else if (status === 400) {
        return { status: 'bad_request', reason: handle_bad_request_reason(resp) };
    } else if (status === 401 || status === 403) {
        return { status: 'wrong_ticket' };
    } else if (is_internal_error(status)) {
        return { status: 'retryable_error' };
    } else {
        throw Error('strange status: ' + status);
    }
};

const saving_message = (status, resp) => {
    if (status === 413) {
        return { status: 'limited' };
    } else if (status === 410) {
        return { status: 'cannot_save' };
    } else {
        return common_codes(status, resp);
    }
};

const sending_message = (status, resp) => {
    if (status === 505) {
        return { status: 'message_saved' };
    } else if (status === 203) {
        return { status: 'from_cache' };
    } else if (status === 402) {
        return { status: 'captcha' };
    } else if (status === 409) {
        return {
            status: 'spam',
            reason: handle_spam_reason(resp),
            banReason: handle_ban_reason(resp)
        };
    } else {
        return saving_message(status, resp);
    }
};

const wrapper = (status, body, handler) => {
    let resp;
    try {
        resp = JSON.parse(body);
    } catch (e) {
        resp = undefined;
    }

    return Object.assign({}, handler(status, resp), { object: resp });
};

const save_draft = (status, body) => {
    return wrapper(status, body, saving_message);
};

const save_template = (status, body) => {
    return wrapper(status, body, saving_message);
};

const send_message = (status, body) => {
    return wrapper(status, body, sending_message);
};

const send_service = (status, body) => {
    return wrapper(status, body, sending_message);
};

const send_delayed = (status, body) => {
    return wrapper(status, body, sending_message);
};

const send_share = (status, body) => {
    return wrapper(status, body, sending_message);
};

const send_undo = (status, body) => {
    return wrapper(status, body, sending_message);
};

const write_attachment = (status, body) => {
    return wrapper(status, body, common_codes);
};

const list_unsubscribe = (status, body) => {
    return wrapper(status, body, common_codes);
};

const cancel_send_undo = (status, body) => {
    return wrapper(status, body, common_codes);
};

const cancel_send_delayed = (status, body) => {
    return wrapper(status, body, common_codes);
};

const limits = (status, body) => {
    return wrapper(status, body, common_codes);
};

const generate_operation_id = (status, body) => {
    return wrapper(status, body, common_codes);
};

module.exports = {
    is_internal_error,
    handle_bad_request_reason,
    handle_spam_reason,
    save_draft,
    save_template,
    send_message,
    send_service,
    send_delayed,
    send_share,
    send_undo,
    write_attachment,
    list_unsubscribe,
    cancel_send_undo,
    cancel_send_delayed,
    limits,
    generate_operation_id
};
