#include <mail/sendbernar/client/include/response.h>
#include <mail/sendbernar/client/include/category.h>
#include <mail/sendbernar/client/include/exception.h>
#include <mail/sendbernar/client/include/internal/result_reflection.h>
#include <mail/sendbernar/composer/include/delivery_helpers.h>
#include <yamail/data/serialization/json_writer.h>


namespace sendbernar::response {

error::Response make_error_resp(mail_errors::error_code ec) {
    const auto& c = ec.category();
    return error::Response{c.name(), c.message(ec.value()), ec.message()};
}

error::LimitedResponse make_limited(mail_errors::error_code ec, const std::string& messageId) {
    const auto& c = ec.category();
    return error::LimitedResponse{
        c.name(),
        c.message(ec.value()),
        ec.message(),
        messageId
    };
}

ResponseStatus toResponseStatus(const mail_errors::error_code& ec) {
    if (ec.category() == getSendbernarCategory()) {
        return toResponseStatus(ErrorResult(ec.value()));
    } else if (ec.category() == getComposeCategory()) {
        return toResponseStatus(ComposeResult(ec.value()));
    } else if (ec.category() == getNwCategory()) {
        return toResponseStatus(DeliveryResult(ec.value()));
    } else {
        return ResponseStatus::internalError;
    }
}

void checkResponseStatus(const mail_errors::error_code& ec, ResponseStatus code, bool failOnLimited) {
    if (!ec) {
        throw Exception("cannot response with error_code without error");
    }

    const bool cannotResponseWithSimpleError = code == ResponseStatus::captchaRequested
            || code == ResponseStatus::spamOrVirusWasFound
            || code == ResponseStatus::saveInsteadOfSending
            || (failOnLimited && code == ResponseStatus::limited)
            || code == ResponseStatus::ok
            || code == ResponseStatus::fromCache;

    if (cannotResponseWithSimpleError) {
        throw Exception("response should not be captchaRequested, spamOrVirusWasFound, saveInsteadOfSending, ok, fromCache" +
                        (failOnLimited ? std::string(", limited") : std::string()));
    }
}

ymod_webserver::codes::code make_code(ResponseStatus status) {
    ymod_webserver::codes::code c = ymod_webserver::codes::internal_server_error;
    switch(status) {
        case ResponseStatus::ok:
            c = ymod_webserver::codes::ok; break;
        case ResponseStatus::badRequest:
            c = ymod_webserver::codes::bad_request; break;
        case ResponseStatus::spamOrVirusWasFound:
            c = ymod_webserver::codes::conflict; break;
        case ResponseStatus::captchaRequested:
            c = ymod_webserver::codes::payment_required; break;
        case ResponseStatus::internalError:
            c = ymod_webserver::codes::internal_server_error; break;
        case ResponseStatus::limited:
            c = ymod_webserver::codes::request_entity_too_large; break;
        case ResponseStatus::saveInsteadOfSending:
            c = ymod_webserver::codes::version_not_supported; break;
        case ResponseStatus::cannotSaveMessage:
            c = ymod_webserver::codes::gone; break;
        case ResponseStatus::wrongMethod:
            c = ymod_webserver::codes::method_not_allowed; break;
        case ResponseStatus::fromCache:
            c = ymod_webserver::codes::non_authoritative_info; break;
    }
    return c;
}

Response::Response(const std::string& data) :
    code(make_code(ResponseStatus::ok)),
    body(data),
    contentType(std::make_pair("text", "plain"))
{}

template<class T>
void Response::initJson(ResponseStatus code_, const T& data) {
    code = make_code(code_);
    body = yamail::data::serialization::toJson(data).str();
    contentType = std::make_pair("application", "json");
}

Response::Response(LimitsResponse response)
    : code(make_code(ResponseStatus::ok))
    , body(yamail::data::serialization::toJson(response).str())
    , contentType(std::make_pair("application", "json"))
{ }

Response::Response(const ComposeDraftResult& response) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(response).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const ComposeMessageResult& response) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(response).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(mail_errors::error_code ec, const std::string& messageId) :
    errorCode(std::move(ec))
{
    ResponseStatus code = toResponseStatus(errorCode);
    checkResponseStatus(errorCode, code, false);

    if (code == ResponseStatus::limited) {
        initJson(code, make_limited(errorCode, messageId));
    } else {
        initJson(code, make_error_resp(errorCode));
    }
}

Response::Response(MessageSentResponse response) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(response).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(MessageSavedResponse response) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(response).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const SaveMessageWithOptionalCaptcha& data) :
    errorCode(make_error(data.result))
{
    ResponseStatus code = toResponseStatus(data.result);

    if (code == ResponseStatus::captchaRequested) {
        initJson(code, error::CaptchaRequested {
            errorCode.category().name(),
            errorCode.category().message(errorCode.value()),
            errorCode.message(), data.response.messageId, data.captcha, data.response.stored
        });
    } else if (code == ResponseStatus::spamOrVirusWasFound) {
        initJson(code, error::SpamOrVirusWasFound {
            errorCode.category().name(),
            errorCode.category().message(errorCode.value()),
            errorCode.message(), data.banReason, data.response.messageId, data.response.stored
        });
    } else {
        initJson(ResponseStatus::saveInsteadOfSending, data.response);
    }
}

Response::Response(const DelayedMessageCallbackResponse& data) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(data).str()),
    contentType(std::make_pair("application", "json"))
{ }

Response::Response(const CallbackResponseError& data)
    : code(data.stopTrying ? ymod_webserver::codes::bad_request : ymod_webserver::codes::internal_server_error)
    , body("{}")
    , contentType(std::make_pair("application", "json"))
{
    if (data.stopTrying) {
        additionalHeaders.add("X-Ya-CallMeBack-Notify-Reject", "yes");
    }
}

Response::Response(const RemindMessageCallbackResponse& data) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(data).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const SendBarbetMessageResponse& data) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(data).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const NoAnswerRemindCallbackResponse& data) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(data).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const ListUnsubscribeResponse& data) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(data).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const GenerateOperationIdResponse& data)
    : code(make_code(ResponseStatus::ok))
    , body(yamail::data::serialization::toJson(data).str())
    , contentType(std::make_pair("application", "json"))
{ }

Response::Response(const MessageSentCacheResponse& data)
    : code(make_code(ResponseStatus::fromCache))
    , body(yamail::data::serialization::toJson(data).str())
    , contentType(std::make_pair("application", "json"))
{ }

Response::Response(mail_errors::error_code ec)
    : errorCode(std::move(ec))
{
    ResponseStatus code = toResponseStatus(errorCode);
    checkResponseStatus(errorCode, code, true);

    initJson(code, make_error_resp(errorCode));
}

Response::Response(const WriteAttachmentResponse& data) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(data).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const CancelSendDelayedResponse& data) :
    code(make_code(ResponseStatus::ok)),
    body(yamail::data::serialization::toJson(data).str()),
    contentType(std::make_pair("application", "json"))
{}

Response::Response(const MessageWithErrorOnSharing& data) :
    code(make_code(toResponseStatus(data.result))),
    body(yamail::data::serialization::toJson(make_error_resp(make_error(data.result))).str()),
    contentType(std::make_pair("application", "json"))
{}

}
