#include <mail/sendbernar/composer/include/attach_helpers.h>
#include <boost/range/combine.hpp>
#include <mail_getter/UTFizer.h>
#include <openssl/sha.h>

namespace sendbernar {

bool mustBeEncodedBase64(const std::string& t) {
    if (t.size() > (64 * 1024)) {
        return true;
    }
    std::size_t last_cr = 0;
    for (size_t i = 0; i < t.size(); ++i) {
        if ((!std::isprint(t[i]) && !std::isspace(t[i]))
                && (static_cast<unsigned char>(t[i]) < 128)) {
            return true;
        }

        if ((i - last_cr + 1) > 998) {
            return true;
        }

        if (t[i] == '\n') {
            last_cr = i + 1;
        }
    }
    return false;
}

std::tuple<MimeParser::Hid, std::string, std::string> AttachHelpers::addAttachmentToMessage(RfcMessage& msg,
                                                                                            std::string filename,
                                                                                            std::string contentType,
                                                                                            const std::string& body) const {
    UTFizer::process(recognizer_, "utf-8", filename);
    std::string::size_type sp = filename.find_last_of("/\\");
    if (sp != std::string::npos) {
        filename = filename.substr(sp + 1);
    }

    MimeType mType(contentType);
    if (mType.isDefaultMimeType()) {
        mType = detector_.detect(filename, body);
    }

    ContentTypeDetector::adjustType(mType);

    return std::make_tuple(msg.addBase64File(mType, body, filename), filename, mType.toString());
}

std::vector<compose::Attachment> AttachHelpers::getStorageAttachments(RfcMessage& m) {
    std::vector<compose::Attachment> res;

    if (storageAttachments_.size() != idsList_.size()) {
        throw std::runtime_error("Size of storageAttachments is " + std::to_string(storageAttachments_.size()) +
                                 ", size of idsList is " + std::to_string(idsList_.size()));
    }

    for (const auto p: boost::combine(storageAttachments_, idsList_)) {
        const mail_getter::AbstractAttachment& attachment = *boost::get<0>(p);
        const std::string& sid = boost::get<1>(p);

        std::string filename = attachment.getFilename();
        std::string contentType = attachment.getContentType();

        std::string attachmentBody = attachment.getBody();
        MimeType mType(contentType);
        if (mType.isDefaultMimeType()) {
            mType = detector_.detect(filename, attachmentBody);
        }
        contentType = mType.toString();
        const auto hash = calcHashAttachment(attachmentBody, filename);

        MimeParser::Hid hid;
        std::tie(hid, filename, contentType) = addAttachmentToMessage(m, filename, contentType, attachmentBody);

        res.emplace_back(compose::Attachment{
            .hid_ = hid.toString(),
            .contentType_ = contentType,
            .fileName_ = filename,
            .id_ = sid,
            .size_ = attachmentBody.size(),
            .hash_ = std::move(hash),
        });
    }

    return res;
}

std::string calcHashAttachment(const std::string& body, const std::string& name) {
    SHA512_CTX context;
    SHA512_Init(&context);

    SHA512_Update(&context, body.data(), body.size());
    SHA512_Update(&context, name.data(), name.size());

    std::string digestBuff(SHA512_DIGEST_LENGTH, 'A');
    SHA512_Final(reinterpret_cast<unsigned char*>(digestBuff.data()), &context);

    return encode_base64(digestBuff);
}

}
