#include <mail/sendbernar/composer/include/common_headers.h>
#include <mail/sendbernar/composer/include/fold_header.h>
#include <butil/datetime/date_utils.h>


namespace sendbernar {

std::string realMailbox(const Account& auth, const std::string& authDomain) {
    const std::string& userDomain = !auth.domain.empty() ? auth.domain : authDomain;
    return EmailHelpers::toString(Email(auth.login, userDomain));
}

std::string nonEmptySubject(const std::string& subj) {
    return subj.empty() ? "No subject" : subj;
}

std::string timeToString(std::time_t now) {
    return DateUtils::rfc2822time(now, "Europe/Moscow");
}

std::string CommonHeaders::realMailbox() const {
    return sendbernar::realMailbox(auth_, authDomain_);
}

std::string CommonHeaders::envelopeFrom() const {
    return EmailHelpers::idnaize(auth_.hasAddress(fromMailbox_) ? fromMailbox_ : realMailbox());
}

std::string CommonHeaders::from() {
    Email from(EmailHelpers::fromString(fromMailbox_));
    from.setDisplayName(rfc2822ns::quote(fromName_));
    return EmailHelpers::toString(from);
}

std::string CommonHeaders::nonEmptySubject() const {
    return ::sendbernar::nonEmptySubject(subject_);
}

CachedComposeResult CommonHeaders::apply(RfcMessage& m, RecipientsHeaders& recipients, const std::string messageId) {
    CachedComposeResult result;

    m.addHeader("Received", "by " + hostname_ + " with HTTP;\r\n\t" + timeToString(now_));

    m.addEmailHeader("From", from());

    const std::string rmb = realMailbox();
    if (fromMailbox_ != rmb) {
        m.addEmailHeader("Envelope-From", rmb);
    }

    result = recipients.apply(m);

    if (!inReplyTo_.empty()) {
        m.addHeader("In-Reply-To", inReplyTo_);
    }

    if (!references_.empty()) {
        m.addHeader("References", foldHeader(references_));
    }

    if (!subject_.empty()) {
        m.addUtf8Header("Subject", subject_);
    }

    m.addHeader("MIME-Version", "1.0");
    m.addHeader("X-Mailer", "Yamail [ http://yandex.ru ] 5.0");
    m.addHeader("Date", DateUtils::rfc2822time(date_, auth_.timezone));

    if (highPriority_) {
        m.addHeader("X-Priority", "1");
    }

    if (!mobileCaller_.empty()) {
        m.addHeader("X-Yandex-Mobile-Caller", mobileCaller_);
    }

    m.addMessageId(messageId);

    if (captchaStatus_ != CheckCaptchaResult::doNotRequired) {
        m.addHeader("X-Yandex-Captcha-Entered", captchaStatus_ == CheckCaptchaResult::good ? "yes" : "bad");
    }

    return result;
}

}
