#include <mail/sendbernar/composer/include/delayed_message_compose.h>
#include <mail/sendbernar/composer/include/common_headers.h>
#include <mail/sendbernar/composer/include/settings.h>
#include <mail/sendbernar/core/include/config.h>


namespace sendbernar {

DelayedMessageComposer::DelayedMessageComposer(const params::CommonParams& params, ContextLogger logger)
    : requestId(params.requestId)
    , logger_(logger)
    , authDomain_(authDomain(params))
{ }

DelayedMessageComposer::NeedToSend DelayedMessageComposer::needToSend(const macs::Mid& mid, MetadataPtr m) {
    try {
        envelope_ = m->getById(mid);
    } catch(const macs::system_error& ex) {
        if (ex.code() == macs::error::noSuchMessage) {
            return NeedToSend(DelayedMessageCallbackResponse());
        } else {
            return NeedToSend(ex.code());
        }
    }

    const auto outboxFid = m->fidBySymbol(macs::Folder::Symbol::outbox);

    if (envelope_.fid() != outboxFid) {
        const auto delayed = m->labelBySymbol(macs::Label::Symbol::delayedMessage_label);
        const auto undo    = m->labelBySymbol(macs::Label::Symbol::undoMessage_label);
        m->unmarkEnvelopes(envelope_.mid(), {delayed, undo});
        return NeedToSend(DelayedMessageCallbackResponse());
    }

    return NeedToSend(ContinueSendingDelayedMessage());
}

compose::SendMessage DelayedMessageComposer::message(const Account& a, const macs::Mid& mid, MetadataPtr m,
                                                     SendConfigurationPtr cfg) {
    const auto mgLogger = getMulcagateLogger(logger_);
    const auto storageService = cfg->mailStorage()->createService(mgLogger, tvm::Ticket(), requestId);
    const auto access = storageService->createMessageAccess(m->getMessageAccessParams(mid), cfg->recognizer());
    const auto attrs = access->getMessageHeaderParsed(rootHid);

    const auto header = [&attrs](const std::string& name) {
        const auto iter = attrs.find(name);
        return iter == attrs.end() ? "" : iter->second;
    };

    const std::string fromEmail = EmailHelpers::fromString(envelope_.from()).addressString();

    recipients_.loadTo(header("to"));
    recipients_.loadCc(header("cc"));
    recipients_.loadBcc(header("bcc"));

    lids_ = envelope_.labels();

    compose::SendMessage message;

    message.date = envelope_.date();
    message.text = boost::make_shared<std::string>(access->getWhole());
    message.envelopeFrom = a.hasAddress(fromEmail) ? fromEmail : realMailbox(a, authDomain_);
    message.realMailbox = realMailbox(a, authDomain_);
    message.messageId = envelope_.rfcId();
    message.nonEmptySubject = sendbernar::nonEmptySubject(envelope_.subject());
    message.hasAttachments = envelope_.hasLabel(m->labelBySymbol(macs::Label::Symbol::attached_label));
    message.recipients = recipients_;
    message.saveToSent = true;

    return message;
}

}
