#include <ctime>
#include <optional>
#include <algorithm>
#include <mail/sendbernar/core/include/recipients_repository.h>
#include <mail/sendbernar/composer/include/recipients_headers.h>
#include <mail/sendbernar/composer/include/rfc_message.h>
#include <mail/sendbernar/core/include/cached_compose_result.h>
#include <mail/sendbernar/core/include/account.h>
#include <mail/sendbernar/client/include/params.h>
#include <mail/sendbernar/core/include/metadata.h>
#include <mail/sendbernar/composer/include/settings.h>
#include <mail/sendbernar/core/include/check_captcha_result.h>

#include <util/generic/typetraits.h>


namespace sendbernar {

std::string timeToString(std::time_t now);
std::string nonEmptySubject(const std::string& subj);
std::string realMailbox(const Account& auth, const std::string& authDomain);

namespace detail {
template<class Params>
inline bool isHighPriority(const Params& p, MetadataPtr metadata) {
    if (!params::normalize(p).lids) {
        return false;
    }

    const auto& lids = *params::normalize(p).lids;
    std::string importantLid = metadata->labelBySymbol(macs::Label::Symbol::important_label).lid();
    return std::find(lids.begin(), lids.end(), importantLid) != lids.end();
}

template<class Params>
inline std::time_t extractSendDate(const Params& params, unsigned additionalTime) {
    if constexpr (std::is_same<Params, params::SendDelayed>::value) {
        return delayedMessageSendDate(params, additionalTime);
    } else {
        return 0;
    }
}

template<class Params>
inline std::string inreplyto(const Params& p) {
    if constexpr (std::is_same<Params, params::SendDelayed>::value) {
        return inreplyto(p.send);
    } else if constexpr (std::is_same<Params, params::SaveTemplate>::value) {
        return "";
    } else if constexpr (std::is_same<decltype(Params().inreplyto), boost::optional<std::string>>::value) {
        return p.inreplyto.get_value_or("");
    } else if constexpr (std::is_same<decltype(Params().inreplyto), boost::optional<params::InReplyTo> >::value) {
        if (p.inreplyto) {
            return p.inreplyto->inreplyto;
        }

        return "";
    } else {
        static_assert(TDependentFalse<Params>, "strange type for inreplyto");
    }
}

template<class Params>
inline std::string references(const Params& p) {
    if constexpr (std::is_same<Params, params::SendDelayed>::value) {
        return p.send.references.get_value_or("");
    } else if constexpr (std::is_same<Params, params::SaveTemplate>::value) {
        return "";
    } else {
        return p.references.get_value_or("");
    }
}

template<class Params>
inline std::string fromMailbox(const Account& auth, LazySettings profile, const Params& p) {
    if (!params::normalize(p).sender.from_mailbox) {
        return profile.get().defaultEmail;
    }

    const std::string fromMailbox = params::normalize(p).sender.from_mailbox.get_value_or("");

    return auth.validateAddress(fromMailbox) ? fromMailbox
                                             : profile.get().defaultEmail;
}

template<class Params>
inline std::string fromName(LazySettings profile, const Params& p) {
    if (params::normalize(p).sender.from_name) {
        return *params::normalize(p).sender.from_name;
    } else {
        return profile.get().fromName;
    }
}

}

static constexpr unsigned MAX_DIFFERENCE_BETWEEN_STD_TIME_AND_PASSED_PARAM = 100;

inline bool validateCurrentTime(std::time_t now, const boost::optional<std::time_t>& currentTime) {
    if (currentTime) {
        if (std::max(*currentTime, now) - std::min(*currentTime, now) < MAX_DIFFERENCE_BETWEEN_STD_TIME_AND_PASSED_PARAM) {
            return true;
        }
    }
    return false;
}

constexpr unsigned MESSAGE_ID_HEADER_MAX_LENGTH = 100;

inline bool validateHeaderMessageId(const std::string& msgId) {
    return msgId.size() > 0
            && msgId.size() < MESSAGE_ID_HEADER_MAX_LENGTH
            && msgId.front() == '<'
            && msgId.back() == '>'
            && msgId.find_first_of("@") == msgId.find_last_of("@")
            && msgId.find_first_of("@") != std::string::npos;
}

class HeaderMessageId {
    std::string messageId_;

public:
    template<class Params>
    HeaderMessageId(const std::string& sourceMidMessageId, const Params& p,
                    std::function<std::string(void)> generate = &generate_message_id)
    {
        if (params::normalize(p).message.message_id &&
            validateHeaderMessageId(*params::normalize(p).message.message_id)) {
            messageId_ = *params::normalize(p).message.message_id;
        } else if (!sourceMidMessageId.empty()) {
            messageId_ = sourceMidMessageId;
        } else {
            messageId_ = generate();
        }
    }

    HeaderMessageId(const HeaderMessageId&) = default;
    HeaderMessageId(HeaderMessageId&&) = default;

    const std::string& get() const {
        return messageId_;
    }
};

class CommonHeaders {
    friend struct CommonHeadersTest;

    std::string hostname_;
    std::time_t now_;
    CheckCaptchaResult captchaStatus_;
    std::string mobileCaller_;
    std::string authDomain_;
    std::string fromMailbox_;
    std::string fromName_;
    std::string inReplyTo_;
    std::string references_;
    std::string subject_;
    std::time_t sendDate_ = 0;
    std::time_t date_ = 0;
    bool highPriority_ = false;
    const Account& auth_;

public:
    std::string from();
    std::string realMailbox() const;
    std::string envelopeFrom() const;
    std::time_t sendDate() const {
        return sendDate_;
    }
    std::time_t date() const {
        return date_;
    }
    std::string nonEmptySubject() const;


    CachedComposeResult apply(RfcMessage& m, RecipientsHeaders& recipients, const std::string messageId);

    template<class Params>
    CommonHeaders(const Account& auth, std::time_t now, CheckCaptchaResult captchaStatus,
                  const std::string& hostname, const std::string& mobileCaller,
                  MetadataPtr metadata, unsigned additionalTime, const std::string& authDomain,
                  LazySettings profile, const Params& p)
        : hostname_(hostname)
        , now_(validateCurrentTime(now, params::normalize(p).message.current_time) ? *params::normalize(p).message.current_time
                                                                                   : now)
        , captchaStatus_(captchaStatus)
        , mobileCaller_(mobileCaller)
        , authDomain_(authDomain)
        , fromMailbox_(detail::fromMailbox(auth, profile, p))
        , fromName_(detail::fromName(profile, p))
        , inReplyTo_(detail::inreplyto(p))
        , references_(detail::references(p))
        , subject_(params::normalize(p).message.subj.get_value_or(""))
        , sendDate_(detail::extractSendDate(p, additionalTime))
        , date_(!sendDate_ ? now_ : sendDate_)
        , highPriority_(detail::isHighPriority(p, metadata))
        , auth_(auth)
    { }

    CommonHeaders(const Account& auth, std::time_t now, const std::string& hostname, const std::string& authDomain, const params::SendShare& p)
        : hostname_(hostname)
        , now_(now)
        , captchaStatus_(CheckCaptchaResult::doNotRequired)
        , mobileCaller_("")
        , authDomain_(authDomain)
        , fromMailbox_(auth.defaultAddress)
        , fromName_(auth.fromName)
        , inReplyTo_(p.inreplyto.get_value_or(""))
        , references_(p.references.get_value_or(""))
        , subject_(p.message.subj.get_value_or(""))
        , sendDate_(0)
        , date_(now)
        , highPriority_(false)
        , auth_(auth)
    { }

    CommonHeaders(const CommonHeaders&) = default;
    CommonHeaders(CommonHeaders&&) = default;
};

}
