#pragma once

#include <list>
#include <string>
#include <butil/butil.h>
#include <time.h>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/array.hpp>
#include <boost/bind.hpp>
#include <boost/xpressive/xpressive.hpp>

// get current unix-time in seconds since epoch plus expiration period (in seconds as well)
inline std::string timeStamp(int expire = 0)
{
    return boost::lexical_cast<std::string>(time(0) + expire);
}

const std::string::size_type maxHeaderLineLength = 800;

struct FoldHeaderStringError : std::runtime_error
{
    FoldHeaderStringError(const std::string &message) :
        std::runtime_error("foldHeader input string error: " + message) {}
};

struct FoldHeaderAlgorithmError : std::logic_error
{
    FoldHeaderAlgorithmError(const std::string &message) :
        std::logic_error("foldHeader algorithm error: " + message) {}
};

template <std::string::size_type treshold>
inline std::string foldHeaderTemplate(const std::string &str)
{
    using namespace boost::xpressive;

    const std::string separator = "\r\n\t";

    const sregex regexp = bos >> (
            *_s >>
            (s1 = (~_s >> !(repeat<0, treshold - 2>(_) >> ~_s))) >>
            (+_s | eos)
        ) | (
            *_s >>
            (s1 = (repeat<0, treshold - 1>(~_s) >> ',')) >>
            *_s
        ) | (
            *_s >>
            (s1 = repeat<treshold, treshold>(~set[_s | ','])) >>
            *_s
        ) | (
            +_s >> eos
        );

    std::list<std::string> blocks;

    for (std::string::const_iterator it = str.begin(); it != str.end();) {
        smatch what;

        if (!regex_search(it, str.end(), what, regexp)) {
            throw FoldHeaderStringError("string doesn't match regexp");
        } else if (what[0].length() == 0) {
            throw FoldHeaderAlgorithmError("empty match result");
        } else if (what[0].length() > str.end() - it) {
            throw FoldHeaderAlgorithmError("match result larger than rest of str");
        }

        blocks.push_back(what[1]);
        it += what[0].length();
    }

    return boost::algorithm::join(blocks, separator);
}

inline std::string foldHeader(const std::string &str)
{
    return foldHeaderTemplate<maxHeaderLineLength>(str);
}
