#pragma once

#include <mail/sendbernar/core/include/metadata.h>
#include <mail/sendbernar/core/include/cached_compose_result.h>
#include <mail/sendbernar/composer/include/inline_handlers.h>
#include <mail/sendbernar/composer/include/sanitizer.h>

#include <mail_getter/sanitizer/sanitizer_response_processor.h>
#include <mail_getter/sanitizer/sanitizer_markup.h>


namespace sendbernar {

struct InlineProcessor {
    struct Response {
        RemoteAttachments attaches;
        std::string body;
    };

    MetadataPtr metadata;
    SanitizerPtr sanitize;
    mail_getter::ServicePtr storageService;
    const Recognizer::Wrapper& recognizer;
    const mail_getter::attach_sid::KeyContainer& keyContainer;
    const InlineClassHandlers& inlineHandlers;
    ContextLogger logger;
    http_getter::TypedClientPtr httpPtr;
    std::string uid;
    std::string host;

    std::optional<Response> process(std::string body, const std::string& messageId);
};

using InlineProcessorPtr = std::shared_ptr<InlineProcessor>;

class SanitizerAndInlineAttaches {
    bool html_;
    std::size_t maxSanitizerRequestSize_;
    InlineProcessor inlineProcessor_;

public:
    template<class Params>
    SanitizerAndInlineAttaches(const Params& p, std::size_t maxSanitizerRequestSize, InlineProcessor inlineProcessor)
        : html_(params::normalize(p).message.html.get_value_or(false))
        , maxSanitizerRequestSize_(maxSanitizerRequestSize)
        , inlineProcessor_(std::move(inlineProcessor))
    { }

    SanitizerAndInlineAttaches(const SanitizerAndInlineAttaches&) = default;
    SanitizerAndInlineAttaches(SanitizerAndInlineAttaches&&) = default;

    CachedComposeResult apply(std::string& body, RemoteAttachments& attaches, const std::string& messageId) {
        if (html_) {
            if (body.size() > maxSanitizerRequestSize_) {
                return ComposeResult::MSG_TOO_BIG;
            }

            if (auto resp = inlineProcessor_.process(std::move(body), messageId); resp != std::nullopt) {
                attaches = std::move(resp->attaches);
                body = std::move(resp->body);
                LOGDOG_(inlineProcessor_.logger, debug, log::body_size=body.size());
            } else {
                return ComposeResult::SANITIZER_CALL_WAS_NOT_SUCCESS;
            }
        }

        return ComposeResult::DONE;
    }
};

}
