#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <mail/sendbernar/composer/include/inline_content.h>
#include <butil/StrUtils/Encoding.h>

namespace sendbernar {

bool InlineContent::parseId(const std::string & classId, std::string & st_id,
        std::string & hid, std::string & mid) {
    std::vector<std::string> items;
    boost::split(items, classId, boost::is_any_of("_"));
    if (items.size() != 3 || std::find(items.begin(), items.end(), "") != items.end()) {
        return false;
    }
    st_id = std::move(items[0]);
    hid = std::move(items[1]);
    mid = std::move(items[2]);
    return true;
}

bool InlineContent::isValidStid(const std::string& st_id, const std::string& mid,
                                InlineHandlerAttributes::Stids& validStids, InlineHandlerAttributes::Stids& invalidStids,
                                MetadataPtr metadata) {
    if (std::find(validStids.begin(), validStids.end(), st_id) != validStids.end()) {
        return true;
    }

    if (std::find(invalidStids.begin(), invalidStids.end(), st_id) != invalidStids.end()) {
        return false;
    }

    try {
        macs::Envelope env = metadata->getById(mid);
        if (env.stid() != st_id) {
            invalidStids.push_back(st_id);
            return false;
        } else {
            validStids.push_back(st_id);
            return true;
        }
    } catch(const macs::system_error&) {
        invalidStids.push_back(st_id);
        return false;
    }
}

MessageAccessPtr InlineContent::getMessageAccess(const std::string& mid, InlineHandlerAttributes::Messages& messages,
                                                 MetadataPtr metadata, mail_getter::ServicePtr mailStorageService,
                                                 const Recognizer::Wrapper& recognizer) {
    if (auto it = messages.find(mid); it == messages.end()) {
        const auto ma = mailStorageService->createMessageAccess(metadata->getMessageAccessParams(mid), recognizer);
        return messages.emplace(mid, ma).first->second;
    } else {
        return it->second;
    }
}

int InlineContent::getAttachment(const std::string & classId,
        RemoteAttachment & att, InlineHandlerAttributes &attr) {
    LOGDOG_(attr.logger, notice, log::where_name="inline_content", log::message=classId);
    std::string st_id, hid, mid;
    if ( !parseId( classId, st_id, hid, mid ) ) {
        return 0;
    }

    if (!isValidStid(st_id, mid, attr.validStids, attr.invalidStids, attr.metadata)) {
        return 0;
    }

    const auto ma = getMessageAccess(mid, attr.messages, attr.metadata, attr.mailStorageService, attr.recognizer);

    auto content = ma->getBody(hid);
    auto hs = ma->getHeaderStruct(hid);

    att.name = getFilename(hs);
    att.disposition = "inline";
    att.encoding = "base64";
    att.cid = hs.cid();
    att.original_url = "http://"+attr.host+"/message_part/"+att.name+"?hid="+hid+"&amp;mid="+st_id+"&amp;name="+att.name;
    Encoding::decode ( hs.encoding(), content );
    att.body = std::move(content);
    return 1;
}

}
