#include <mail/sendbernar/composer/include/mail_compose.h>
#include <mail/sendbernar/composer/include/sanitizer.h>
#include <mail_getter/vdirect/secure_vdirect.h>
#include <mail/sendbernar/core/include/composed_message.h>
#include <mail/sendbernar/core/include/cached_compose_result.h>
#include <mail/sendbernar/composer/include/inline_classes.h>
#include <mail/sendbernar/core/include/recipients_repository.h>
#include <mail/sendbernar/composer/include/inline_processor.h>
#include <mail/sendbernar/composer/include/rfc_message.h>
#include <mail/sendbernar/composer/include/common_headers.h>
#include <mail/sendbernar/composer/include/recipients_headers.h>
#include <mail/sendbernar/composer/include/body_and_attaches.h>
#include <mail/sendbernar/composer/include/has_attaches.h>
#include <mail/sendbernar/composer/include/sids_attachments.h>
#include <mail/sendbernar/composer/include/parts_json.h>
#include <mail/sendbernar/composer/include/mids_attaches.h>
#include <mail/sendbernar/core/include/config.h>
#include <mail/sendbernar/composer/include/smiles.h>
#include <mail/sendbernar/composer/include/inline_content.h>
#include <mail/sendbernar/composer/include/inline_from_verstka.h>
#include <mail_getter/UTFizer.h>

#include <util/generic/typetraits.h>

namespace sendbernar {

namespace {

bool isMobileCaller(const ComposeConfigPtr& conf, const std::string& caller) {
    const auto& callers = conf->mobileCallers;
    return std::find(callers.begin(), callers.end(), caller) != callers.end();
}

}

struct ScopeExit {
    pa::stimer_t krotCompose;
    std::string realMailbox;
    ContextLogger& logger;

    ScopeExit(const std::string& r, ContextLogger& l)
        : realMailbox(r)
        , logger(l)
    {
        krotCompose.start();
    }

    ~ScopeExit() {
        if (krotCompose.stop() > 2000) {
            LOGDOG_(logger, warning, log::message="composing of mail for " + realMailbox
                    + " has taken " + std::to_string(krotCompose.shot() / 1000.0) + " seconds");
        }
    }
};

mail_getter::ServicePtr MailCompose::servicePtr() const {
    return conf_->mailStorage->createService(getMulcagateLogger(logger_), tvm::Ticket(), common_.requestId);
}

template<class Params>
CommonHeaders MailCompose::makeCommonHeaders(const Params& p, CheckCaptchaResult captchaStatus) {
    const std::string mobileCallerForCompose = isMobileCaller(conf_, common_.caller) ? common_.caller : "";
    return CommonHeaders(account_, std::time(0), captchaStatus, getHostName(), mobileCallerForCompose,
                         metadata_, conf_->reminders.sendUndoAdditionalTime, authDomain(common_),
                         profile_, p);
}

template<>
CommonHeaders MailCompose::makeCommonHeaders(const params::SendShare& p, CheckCaptchaResult) {
    if (!additionalAccount_) {
        throw std::logic_error("empty additionalAccount");
    }

    return CommonHeaders(*additionalAccount_, std::time(0), getHostName(), authDomain(common_), p);
}

template<class Params>
BodyAndAttaches MailCompose::makeBodyAndAttaches(const Params& p) {
    return BodyAndAttaches(p, conf_->attachmentsMaxSize, conf_->diskAttachesTitles.find(account_.language), *conf_->contentTypeDetector,
                           diskAttaches_, conf_->recognizer, logger_);
}

template<class Params>
SidsAttachments MailCompose::makeSidsAttachments(const Params& p) {
    return SidsAttachments(p, conf_->keyContainer, servicePtr(), *conf_->contentTypeDetector,
                           conf_->recognizer, logger_);
}

template<class Params>
PartsJsonAttachments MailCompose::makePartsJsonAttachments(const Params& p) {
    return PartsJsonAttachments(p, servicePtr(), *conf_->contentTypeDetector, conf_->recognizer,
                                metadata_, conf_->attachmentsMaxSize, logger_);
}

template<class Params>
MidsAttaches MailCompose::makeMidsAttaches(const Params& p) {
    return MidsAttaches(p, metadata_, servicePtr(), conf_->recognizer, conf_->attachmentsMaxSize, logger_);
}

template<class Params, class Handlers>
SanitizerAndInlineAttaches MailCompose::makeSanitizerAndInlineAttaches(const Params& p, const Handlers& handlers) {
    auto sanitizer = makeSanitizer(common_.uid, common_.requestId, conf_->sanitizer, httpPtr_);

    return SanitizerAndInlineAttaches(p, conf_->sanitizer.maxRequestSize, InlineProcessor {
        metadata_, sanitizer,
        servicePtr(),
        conf_->recognizer, conf_->keyContainer, handlers,
        logger_, httpPtr_,
        common_.uid, common_.originalHost
    });
}

VdirectPtr MailCompose::makeVdirect() {
    auto uidProvider = std::make_unique<vdirect::UidHashProvider>(conf_->vdirect.keysStorage, account_.uid);
    return boost::make_shared<SecureVdirect>(SecureVdirect::Params(true, authDomain(common_),
                                                                   conf_->vdirect.uidScript),
                                             std::move(uidProvider));
}

CachedComposeResult checkLimits(const std::string& messageText, std::size_t messageMaxSize) {
    const auto newLineCount = std::count(messageText.begin(), messageText.end(), '\n');
    const auto carriageReturnCount = std::count(messageText.begin(), messageText.end(), '\r');
    const auto msgSize = static_cast<long>(messageText.size()) + newLineCount - carriageReturnCount;

    return static_cast<std::size_t>(msgSize) < messageMaxSize ? ComposeResult::DONE
                                                              : ComposeResult::MSG_TOO_BIG;
}

LimitedResult createLimits(const SmtpLimits& limits, const RecipientsRepository& repo, std::size_t msgSize) {
    LimitedResult limited;
    for(const Email& email : repo.toVector()) {
        if (const auto l = limits.find(email.domain()); l != limits.end()) {
            if (static_cast<unsigned long long>(msgSize) > l->second) {
                limited.push_back({
                    .login = email.local(),
                    .domain = email.domain(),
                    .limit = l->second,
                });
            }
        }
    }

    return limited;
}

CachedComposeResult checkAttachmentsSize(std::size_t totalAttachmentSize, std::size_t attachmentsMaxSize) {
    return totalAttachmentSize > attachmentsMaxSize ? ComposeResult::ATTACHMENT_TOO_BIG
                                                    : ComposeResult::DONE;
}

template<class Result, class Params>
compose::Expected<Result> MailCompose::composeImpl(const Params& params, CheckCaptchaResult captchaStatus) {
    InlineClassHandlers handlers = {
        std::make_shared<Smiles>(conf_->smiles.endpoint, conf_->smiles.host),
        std::make_shared<InlineContent>(),
        std::make_shared<InlineFromVerstka>()
    };

    VdirectPtr vdirect = makeVdirect();
    RecipientsHeaders recipients(conf_->domainsNotToLowercase, conf_->maxRecipients, params);
    boost::shared_ptr<std::string> body = boost::make_shared<std::string>(sendbernar::params::normalize(params).message.text.get_value_or(""));
    CommonHeaders commonHeaders = makeCommonHeaders(params, captchaStatus);
    HeaderMessageId messageId(sourceMidMessageId_, params);
    BodyAndAttaches bodyAndAttaches = makeBodyAndAttaches(params);
    SimpleBody simpleBody(params);
    HasAttaches hasAttaches(params, bodyAndAttaches);
    SidsAttachments sidsAttachments = makeSidsAttachments(params);
    PartsJsonAttachments partsJson = makePartsJsonAttachments(params);
    MidsAttaches midsAttaches = makeMidsAttaches(params);
    SanitizerAndInlineAttaches sanitizerAndInlineAttaches = makeSanitizerAndInlineAttaches(params, handlers);
    std::vector<compose::Attachment> attachments;
    bool hasAnyAttachment = false;

    ScopeExit exit(commonHeaders.realMailbox(), logger_);

#define SuccessExpected(...) compose::Expected<Result>(Result __VA_ARGS__ )
#define ApplyOrReturnError(...) \
    if (auto res = __VA_ARGS__; res != ComposeResult::DONE) {\
        return compose::Expected<Result>(yamail::unexpected_type<compose::Error>({ \
            .result = std::move(res), \
            .messageId = std::move(messageId.get()) \
        })); \
    }

    std::size_t totalAttachmentSize = 0;
    RemoteAttachments remoteAtts;
    RfcMessage m;

    UTFizer::process(conf_->recognizer, "", *body);
    vdirect->unwrap(*body);

    ApplyOrReturnError(sanitizerAndInlineAttaches.apply(*body, remoteAtts, messageId.get()));
    ApplyOrReturnError(commonHeaders.apply(m, recipients, messageId.get()));

    hasAnyAttachment = hasAttaches.hasAttaches(remoteAtts);
    if (!hasAnyAttachment && !body->empty()) {
        ApplyOrReturnError(simpleBody.apply(m, *body));
    } else {
        ApplyOrReturnError(bodyAndAttaches.apply(m, *body, remoteAtts, totalAttachmentSize));
        ApplyOrReturnError(partsJson.apply(m, attachments, totalAttachmentSize, remoteAtts));
        ApplyOrReturnError(midsAttaches.apply(m, totalAttachmentSize));
        ApplyOrReturnError(sidsAttachments.apply(m, attachments, totalAttachmentSize));
        ApplyOrReturnError(checkAttachmentsSize(totalAttachmentSize, conf_->attachmentsMaxSize));
    }

    *body = m.text();

    ApplyOrReturnError(checkLimits(*body, conf_->messageMaxSize));

    if constexpr (std::is_same<Result, compose::SaveDraftOrTemplate>::value) {
        return SuccessExpected({
            .messageId = messageId.get(),
            .attachments = std::move(attachments),
            .realMailbox = commonHeaders.realMailbox(),
            .text = body,
        });
    } else if constexpr (std::is_same<Result, compose::SendDelayed>::value) {
        return SuccessExpected({
            .recipients = recipients.recipients(),
            .limited_ = createLimits(conf_->smtplimits, recipients.recipients(), body->size()),
            .sendDate = commonHeaders.sendDate(),
            .realMailbox = commonHeaders.realMailbox(),
            .text = body,
            .nonEmptySubject = commonHeaders.nonEmptySubject(),
            .messageId = messageId.get(),
            .date = commonHeaders.date(),
            .forwardedMids = midsAttaches.mids(),
            .attachments = std::move(attachments),
        });
    } else if constexpr (std::is_same<Result, compose::SendMessage>::value) {
        return SuccessExpected({
            .recipients = recipients.recipients(),
            .limited_ = createLimits(conf_->smtplimits, recipients.recipients(), body->size()),
            .messageId = messageId.get(),
            .hasAttachments = hasAnyAttachment,
            .saveToSent = true,
            .envelopeFrom = commonHeaders.envelopeFrom(),
            .realMailbox = commonHeaders.realMailbox(),
            .text = body,
            .attachments = std::move(attachments),
            .nonEmptySubject = commonHeaders.nonEmptySubject(),
            .date = commonHeaders.date(),
            .forwardedMids = midsAttaches.mids(),
        });
    } else if constexpr (std::is_same<Result, compose::ComposeMessage>::value) {
        return SuccessExpected({
            .envelopeFrom = commonHeaders.envelopeFrom(),
            .text = body,
            .recipients = recipients.recipients(),
        });
    } else if constexpr (std::is_same<Result, compose::ComposeDraft>::value) {
        return SuccessExpected({
            .attachments = std::move(attachments),
            .envelopeFrom = commonHeaders.envelopeFrom(),
            .text = body,
        });
    } else {
        static_assert(TDependentFalse<Result>, "missing overload");
    }

#undef ApplyOrReturnError
#undef SuccessExpected
}

compose::Expected<compose::SaveDraftOrTemplate> MailCompose::compose(const params::SaveDraft& params) {
    return composeImpl<compose::SaveDraftOrTemplate>(params, CheckCaptchaResult::doNotRequired);
}

compose::Expected<compose::SaveDraftOrTemplate> MailCompose::compose(const params::SaveTemplate& params) {
    return composeImpl<compose::SaveDraftOrTemplate>(params, CheckCaptchaResult::doNotRequired);
}

compose::Expected<compose::SendDelayed> MailCompose::compose(const params::SendDelayed& params, CheckCaptchaResult captchaStatus) {
    return composeImpl<compose::SendDelayed>(params, captchaStatus);
}

compose::Expected<compose::SendMessage> MailCompose::compose(const params::SendMessage& params, CheckCaptchaResult captchaStatus) {
    return composeImpl<compose::SendMessage>(params, captchaStatus);
}

compose::Expected<compose::SendMessage> MailCompose::compose(const params::SendShare& params) {
    return composeImpl<compose::SendMessage>(params, CheckCaptchaResult::doNotRequired);
}

compose::Expected<compose::ComposeMessage> MailCompose::compose(const params::ComposeMessage& params) {
    return composeImpl<compose::ComposeMessage>(params, CheckCaptchaResult::doNotRequired);
}

compose::Expected<compose::ComposeDraft> MailCompose::compose(const params::ComposeDraft& params) {
    return composeImpl<compose::ComposeDraft>(params, CheckCaptchaResult::doNotRequired);
}

}
