#include <mail/sendbernar/composer/include/sids_attachments.h>
#include <boost/range/combine.hpp>
#include <mail_getter/UTFizer.h>
#include <mail/sendbernar/composer/include/attach_helpers.h>

namespace sendbernar {

CachedComposeResult SidsAttachments::apply(RfcMessage& m, std::vector<compose::Attachment>& allAttachments,
                                           std::size_t& totalAttachmentsSize) {
    if (!sids_.empty()) {
        std::vector<macs::Stid> ids;
        try {
            boost::transform(sids_, std::back_inserter(ids), [this] (const std::string& sid) {
                return this->decryptId(sid);
            });
        } catch (const std::exception& ex) {
            LOGDOG_(logger_, error,
                    log::message="Can't decrypt attachment id", log::exception=ex);
            return ComposeResult::ATTACHMENT_STORAGE_ERROR;
        }

        int errorCode;
        VectorOfAttachments storageAttachments;
        std::tie(errorCode, storageAttachments) = getIds(ids);

        if (errorCode != 0) {
            LOGDOG_(logger_, error,
                    log::message="Can't get attachments from the storage. ErrorCode " + std::to_string(errorCode));
            return ComposeResult::ATTACHMENT_STORAGE_ERROR;
        }

        if (sids_.size() != storageAttachments.size()) {
            LOGDOG_(logger_, error,
                    log::message="Invalid attachments count. Expected "
                    + std::to_string(sids_.size()) + ", but was " + std::to_string(storageAttachments.size()));
            return ComposeResult::ATTACHMENT_STORAGE_ERROR;
        }

        for (compose::Attachment attach: getStorageAttachments(m, storageAttachments)) {
            totalAttachmentsSize += attach.size_;
            LOGDOG_(this->logger_, debug, log::attach_size = attach.size_);
            allAttachments.emplace_back(std::move(attach));
        }
    }

    return ComposeResult::DONE;
}

macs::Stid SidsAttachments::decryptId(const std::string& sid) const {
    return mail_getter::decryptId(sid, keyContainer_);
}

std::pair<int, SidsAttachments::VectorOfAttachments> SidsAttachments::getIds(const std::vector<macs::Stid>& ids) const {
    VectorOfAttachments storageAttachments;
    const int errorCode = storageService_->createAttachmentStorage("")->get(ids, storageAttachments);

    return std::make_pair(errorCode, std::move(storageAttachments));
}

std::vector<compose::Attachment>
SidsAttachments::getStorageAttachments(RfcMessage& m, const VectorOfAttachments& attachments) const {
    return AttachHelpers(detector_, recognizer_, attachments, sids_).getStorageAttachments(m);
}

}
