#include <mail/sendbernar/composer/include/system_mail_composer.h>
#include <butil/datetime/date_utils.h>
#include <mail/sendbernar/composer/include/attach_helpers.h>
#include <butil/StrUtils/Iconv.h>
#include <boost/smart_ptr/make_shared.hpp>

namespace sendbernar {

bool tryRecode(const std::string& desired_charset, std::string& text, std::string& charset)
{
    charset = "utf-8";
    if (desired_charset != "utf-8") {
        Iconv cnv("utf-8", desired_charset.c_str(), false);
        std::string new_text;
        if (cnv.recode(text, new_text) != ICONV_OK)
            return false;
        text = new_text;
        charset = desired_charset;
    }
    return true;
}

SystemMailComposer::Expected SystemMailComposer::compose() {
    CachedComposeResult result;

    message m("US-ASCII");

    addReceivedHeader(m);

    if ( !addEmailHeader(m, "From", from()) ) {
        result = ComposeResult::CHARSET_INVALID;
    }

    result = setRecipients(m, to(), cc(), bcc());

    if (!inReplyTo().empty()) {
        m.add_header("In-Reply-To", inReplyTo());
    }

    if (!subject().empty()) {
        if (!addHeader(m, "Subject", subject())) {
            result = ComposeResult::CHARSET_INVALID;
        }
    }

    addCommonHeaders(m);
    addDateHeader(m);

    if (!body().empty()) {
        result = addBody(m, body(), subtype());
    }

    appendMessageText(m);

    if (!checkLimits()) {
        result = ComposeResult::MSG_TOO_BIG;
    }

    if (!result.succeeded()) {
        return Expected(yamail::unexpected_type<ComposeResult>(result.value_));
    }

    compose::SystemMessage msg;

    msg.text = boost::make_shared<std::string>(std::move(messageText_));
    msg.envelopeFrom = EmailHelpers::fromString(fromMailbox_).addressString();

    return msg;
}

const std::string& SystemMailComposer::getMessage() const {
    return messageText_;
}

void SystemMailComposer::appendMessageText(message &m) {
    messageText_ += m.text();
}

std::string SystemMailComposer::from() {
    Email from(EmailHelpers::fromString(fromMailbox_));
    from.setDisplayName(rfc2822ns::quote(fromName_));
    return EmailHelpers::toString(from);
}

std::string SystemMailComposer::messageId() {
    if (!messageId_.empty()) {
        return messageId_;
    }
    messageId_ = generate_message_id();
    return messageId_;
}

bool SystemMailComposer::addEmailHeader(message &m, const std::string &name, std::string value) {
    std::string charset;
    if (!tryRecode(desiredCharset(), value, charset) && strictCharset()) {
        return false;
    }
    m.add_email_header(name, value, charset);
    return true;
}

void SystemMailComposer::addHeader(const std::string &header) {
    messageText_ = header + "\r\n" + messageText_;
}

bool SystemMailComposer::addHeader(message &m, const std::string &name, std::string value) {
    std::string charset;
    if (!tryRecode(desiredCharset(), value, charset) && strictCharset()) {
        return false;
    }
    m.add_header(name, value, charset);
    return true;
}

void SystemMailComposer::addCommonHeaders(message &m) {
    m.add_header("MIME-Version", "1.0");
    m.add_header("Message-Id", messageId());
    m.add_header("X-Mailer","Yamail [ http://yandex.ru ] 5.0");
    m.add_header("X-Yandex-Spam","1");
}

void SystemMailComposer::addReceivedHeader(message &m) {
    m.add_header("Received", "by " + getHostName() + " with HTTP;\n\t");
}

std::string SystemMailComposer::getReceivedDate() {
        time_t currentTime = time(0);
        char time_buf[128];
        tm currentLocalTime( DateUtils::tmDefault() );
        diet_strftime( time_buf, sizeof(time_buf) - 1, "%a, %d %b %Y %H:%M:%S %z",
                diet_localtime_r(&currentTime, &currentLocalTime) );
        return time_buf;
}

void SystemMailComposer::addDateHeader(message &m) {
    m.add_time(time(0));
}

CachedComposeResult SystemMailComposer::addBody(message &m,
        const std::string &text, const std::string &subtype) {
    CachedComposeResult result;
    std::string body = text;
    std::string charset;
    if (!tryRecode(desiredCharset(), body, charset) && strictCharset()) {
        result = ComposeResult::CHARSET_INVALID;
    }

    std::string cte("7bit");
    if (have_8bit(body)) {
        cte="8bit";
        m.set_charset(charset);
    }
    if (mustBeEncodedBase64(body)) {
        cte="base64";
    }
    m.add_body(body, cte);
    m.add_ctype("text", subtype, "");
    return result;
}

CachedComposeResult SystemMailComposer::setRecipients(message &m, const std::string &to,
        const std::string &cc, const std::string &bcc) {
    CachedComposeResult result;

    try {
        recipients_.loadTo(to);
    } catch(...) {
        result = ComposeResult::TO_INVALID;
    }
    try {
        recipients_.loadCc(cc);
    } catch(...) {
        result = ComposeResult::CC_INVALID;
    }
    try {
        recipients_.loadBcc(bcc);
    } catch(...) {
        result = ComposeResult::BCC_INVALID;
    }

    if (recipients_.hasToRecipients()) {
        std::string fieldText =  recipients_.genToString();
        if (!addEmailHeader(m, "To", fieldText)) {
            result = ComposeResult::CHARSET_INVALID;
        }
    }
    if (recipients_.hasCcRecipients()) {
        std::string fieldText = recipients_.genCcString();
        if (!addEmailHeader(m, "Cc", fieldText)) {
            result = ComposeResult::CHARSET_INVALID;
        }
    }

    if (recipients_.hasBccRecipients()) {
        std::string fieldText = recipients_.genBccString();
        if (!addEmailHeader(m, "Bcc", fieldText)) {
            result = ComposeResult::CHARSET_INVALID;
        }
    }

    if (!recipients_.hasRecipients()) {
        result = ComposeResult::TO_CC_BCC_EMPTY;
    } else if (recipients_.isLimitExeeded()) {
        result = ComposeResult::MAX_EMAIL_ADDR_REACHED;
    }
    recipients_.getAsVector(rcpt_);
    return result;
}

bool SystemMailComposer::checkLimits() {
    const auto newLineCount = std::count(getMessage().begin(), getMessage().end(), '\n');
    const auto carriageReturnCount = std::count(getMessage().begin(), getMessage().end(), '\r');
    const auto msgSize = static_cast<long>(getMessage().size()) + newLineCount - carriageReturnCount;

    if (msgSize >= static_cast<long>(messageMaxSize())) {
        return false;
    }
    return true;
}

}
