#include <mail/sendbernar/core/include/configuration.h>
#include <user_journal/service_factory.h>

#include <mail/http_getter/client/include/endpoint_reflection.h>

#include <mail/sendbernar/core/include/sharpei.h>

#include <boost/algorithm/string/split.hpp>
#include <yamail/data/deserialization/ptree_reader.h>
#include <mail_getter/vdirect/secure_vdirect.h>
#include <mail_getter/vdirect/dummy_vdirect.h>
#include <mail_getter/attach_sid_keys.h>

#include <chrono>

#include <mail/callmeback/client/config_reflection.h>
#include <mail/webmail/mulcagate_client/mulca_client.h>


BOOST_FUSION_ADAPT_STRUCT(sendbernar::SendConfiguration::Smtp,
    (http_getter::TypedEndpoint, save)
    (http_getter::TypedEndpoint, save_delayed)
    (http_getter::TypedEndpoint, send)
    (http_getter::TypedEndpoint, system_send)
    (http_getter::TypedEndpoint, delayed_callback_send)
)

BOOST_FUSION_ADAPT_STRUCT(sendbernar::SendConfiguration::Captcha,
    (http_getter::TypedEndpoint, check)
    (http_getter::TypedEndpoint, generate)
    (std::string, type)
    (std::string, strongType)
)


#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-parameter"

namespace sendbernar {

std::time_t delayedMessageSendDate(const params::SendDelayed& params, unsigned additionalTime,
                                   std::function<std::time_t(void)> now) {
    return additionalTime + (params.relative ? (now() + params.send_time) : params.send_time);
}

inline auto loadKeysStorageFromFile(const std::string& name) {
    std::ifstream input(name);
    if (!input.is_open()) {
        throw std::runtime_error(std::string("cannot open file") + name);
    }
    return vdirect::KeysStorage(input);
}

}

#define AS_IS(TYPE, NAME)                       (NAME, TYPE, TYPE, obj.NAME, obj.NAME = val)
#define CHANGE_NAME(TYPE, CONF_NAME, NAME)      (CONF_NAME, TYPE, TYPE, obj.NAME, obj.NAME = val)
#define REMINDER_TEMPLATE(CONF_NAME, NAME)      (CONF_NAME, std::string, std::string, std::string(), obj.NAME = sendbernar::createRemindTemplates(val))
#define READ_KEYS_STORAGE()                     (keys_path, std::string, std::string, std::string(), \
    obj.keysStorage = sendbernar::loadKeysStorageFromFile(val);)

YREFLECTION_ADAPT_ADT(sendbernar::SanitizerConfiguration,
    AS_IS(http_getter::TypedEndpoint, endpoint)
    CHANGE_NAME(size_t, max_request_size, maxRequestSize)
)

YREFLECTION_ADAPT_ADT(sendbernar::RemindersConfiguration,
    CHANGE_NAME(callmeback::ClientConfig, client_configuration, clientConfiguration)
    CHANGE_NAME(unsigned, noanswer_remind_period_sc, noAnswerRemindPeriod)
    CHANGE_NAME(unsigned, send_undo_additional_time_s, sendUndoAdditionalTime)
    REMINDER_TEMPLATE(no_answer_templates_path, noAnswerTemplates)
    REMINDER_TEMPLATE(barbet_create_failed_templates_path, barbetCreateFailedTemplates)
    REMINDER_TEMPLATE(barbet_restore_failed_templates_path, barbetRestoreFailedTemplates)
)

YREFLECTION_ADAPT_ADT(sendbernar::Vdirect,
    CHANGE_NAME(std::string, uid_script, uidScript)
    READ_KEYS_STORAGE()
)

YREFLECTION_ADAPT_ADT(sendbernar::Disk,
    AS_IS(std::string, host)
    CHANGE_NAME(std::string, preview_host, previewHost)
)

YREFLECTION_ADAPT_ADT(sendbernar::SmilesConfiguration,
    AS_IS(http_getter::TypedEndpoint, endpoint)
    AS_IS(std::string, host)
)

#undef READ_KEYS_STORAGE
#undef REMINDER_TEMPLATE
#undef CHANGE_NAME
#undef AS_IS

#pragma GCC diagnostic pop


namespace sendbernar {

namespace {

std::shared_ptr<ContentTypeDetector> initContentTypeDetector(const yplatform::ptree& node) {
    const std::string types = node.get<std::string>("mime");
    const std::string issues = node.get<std::string>("magic.issues");
    const std::string file = node.get<std::string>("magic.file");

    return std::make_shared<ContentTypeDetector>(types, issues, file);
}

std::vector<std::string> initMobileCallers(const yplatform::ptree& node) {
    std::vector<std::string> ret;

    for (const auto& ch : node.get_child("mobile_callers")) {
        ret.push_back(ch.second.get_value<std::string>());
    }

    return ret;
}

template<class T>
T fromChildPtree(const std::string& name, const yplatform::ptree& cfg, ModuleLogger& logger) {
    try {
        return yamail::data::deserialization::fromPtree<T>(cfg.get_child(name));
    } catch (const std::exception& ex) {
        LOGDOG_(logger, error, log::message="an exception while parsing node '" + name +"'", log::exception=ex);
        throw;
    }
}

SmtpLimits initSmtpLimits(const std::string& filename, const ModuleLogger& logger) {
    bool reported = false;
    SmtpLimits limits;
    std::ifstream is(filename.c_str());
    while(is) {
        std::string buffer;
        getline(is, buffer);
        if (is) {
            std::string::size_type pos = buffer.find_first_of(" \t");
            if (pos == std::string::npos) continue;
            std::string domain = buffer.substr(0, pos);
            std::string::size_type pos2 = buffer.find_first_not_of(" \t", pos);
            if (pos2 == std::string::npos) continue;
            unsigned long long limit = boost::lexical_cast<unsigned long long>(buffer.substr(pos2, buffer.length() - pos2));
            if (limit) {
                limits.insert(make_pair(domain, limit));
            } else {
                if (!reported) {
                    reported = true;
                    LOGDOG_(logger, notice, log::message="problem with limits file " + filename);
                }
            }
        }
    }

    return limits;
}

Recognizer::WrapperPtr initRecognizer(const yplatform::ptree& tree) {
    std::string languageDict = tree.get<std::string>("recognizer.language_dict");
    std::string languageWeights = tree.get<std::string>("recognizer.language_weights");
    std::string encodingDict = tree.get<std::string>("recognizer.encoding_dict");
    return Recognizer::create(languageDict.c_str(), languageWeights.c_str(), encodingDict.c_str());
}

std::set<std::string> initDomains(const yplatform::ptree& tree) {
    const std::string str = tree.get<std::string>("space_separated_domains_not_to_be_lowercased");
    std::set<std::string> domains;

    boost::split(domains, str,
                 boost::is_any_of(" "));
    return domains;
}

}

SendConfiguration::SendConfiguration(const http_getter::TypedModulePtr& httpGetter,
                                     const yplatform::ptree& cfg, ModuleLogger& logger)
    : reminders_(fromChildPtree<RemindersConfiguration>("reminders", cfg, logger))
    , diskAttachesTitles_(cfg.get<std::string>("disk_attaches_title_templates_path"))
    , captcha_(fromChildPtree<SendConfiguration::Captcha>("captcha", cfg, logger))
    , vdirect_(fromChildPtree<Vdirect>("vdirect", cfg, logger))
    , wmiYplatformServer_(cfg.get<std::string>("callback_host"))
    , wmiServer_(cfg.get<std::string>("wmi_server"))
    , smtplimits_(initSmtpLimits(cfg.get<std::string>("smtplimit"), logger))
    , sanitizer_(fromChildPtree<SanitizerConfiguration>("sanitizer", cfg, logger))
    , sharpei_(fromChildPtree<SendConfiguration::Sharpei>("sharpei", cfg, logger))
    , mailStorage_(mail_getter::initMailStorage(cfg.get_child("mulcagate"), httpGetter))
    , smiles_(fromChildPtree<SmilesConfiguration>("inline_class_handlers.smiles", cfg, logger))
    , attachmentsMaxSize_(cfg.get<std::size_t>("max_attachments_size"))
    , messageMaxSize_(cfg.get<std::size_t>("message_max_size"))
    , maxRecipients_(cfg.get<unsigned>("max_recipients"))
    , limitsFactor_(cfg.get<double>("limits_factor"))
    , blackBox_(fromChildPtree<SendConfiguration::BlackBox>("blackbox", cfg, logger))
    , contentTypeDetector_(initContentTypeDetector(cfg.get_child("content_type_detector")))
    , nwsmtp_(fromChildPtree<SendConfiguration::NwSmtp>("nwsmtp", cfg, logger))
    , mobileCallers_(initMobileCallers(cfg))
    , keyContainer_(mail_getter::attach_sid::parseKeyContainer(cfg.get_child("attach_sid_keys")))
    , keys_(mail_getter::attach_sid::initKeys(cfg.get_child("attach_sid_keys"), keyContainer_))
    , disk_(fromChildPtree<Disk>("disk", cfg, logger))
    , domainsNotToLowercase_(initDomains(cfg))
    , recognizer_(initRecognizer(cfg))
{}

}
