#pragma once

#include <iterator>
#include <vector>
#include <set>
#include <map>
#include <functional>
#include <butil/email/email.h>
#include <butil/email/helpers.h>


namespace sendbernar {

template<typename Out,typename F>
void transformVec(const EmailVec &vec,Out ins, F func)
{
    std::transform(vec.begin(), vec.end(), ins, boost::bind(func,_1));
}

class Recipient {
private:
    Email email_;
    bool isInternal_;

public:
    Recipient(const Email& email = Email(), bool isInternal = false)
        : email_(email)
        , isInternal_(isInternal)
    {}

    const Email& email() const {
        return email_;
    }

    bool isInternal() const { return isInternal_; }
    void setIsInternal(bool value) { isInternal_ = value; }
};

typedef std::vector<Recipient> Recipients;

struct RecipientsRepository {
    public:
        enum { maxRecipientsUnspecified = 0 };

        RecipientsRepository(size_t maxRecipients = maxRecipientsUnspecified)
        : maxRecipients(maxRecipients)
        {}

        RecipientsRepository(const std::set<std::string>& addressesNotToBeLowercased,
                             size_t maxRecipients = maxRecipientsUnspecified)
        : maxRecipients(maxRecipients)
        , addressesNotToBeLowercased_(addressesNotToBeLowercased)
        {}

        RecipientsRepository(const std::string& to, const std::string& cc,
                             const std::string& bcc, const std::set<std::string>& addressesNotToBeLowercased,
                             size_t maxRecipients = maxRecipientsUnspecified)
        : maxRecipients(maxRecipients)
        , addressesNotToBeLowercased_(addressesNotToBeLowercased)
        {
            loadTo(to);
            loadCc(cc);
            loadBcc(bcc);
        }

    public:
        void loadTo(const std::string & str)
        {
            fromString(str,inserter(to_,to_.end()));
        }

        void loadCc(const std::string & str)
        {
            fromString(str,inserter(cc_,cc_.end()));
        }

        void loadBcc(const std::string & str)
        {
            fromString(str,inserter(bcc_,bcc_.end()));
        }

        bool hasToRecipients() const
        {
            return !to_.empty();
        }

        bool hasCcRecipients() const
        {
            return !cc_.empty();
        }

        bool hasBccRecipients() const
        {
            return !bcc_.empty();
        }

        void getDomains(std::set<std::string>& domains) const;

        void getAsVector(Recipients &recipients) const;

        EmailVec toVector() const;

        bool hasRecipients() const
        {
            return hasToRecipients() || hasCcRecipients() || hasBccRecipients();
        }

        std::string genToString() const
        {
            return EmailHelpers::toString(to_);
        }
        std::string genCcString() const
        {
            return EmailHelpers::toString(cc_);
        }
        std::string genBccString() const
        {
            return EmailHelpers::toString(bcc_);
        }

        void setMaxRecipients (const size_t maxRecipients)
        {
            this->maxRecipients = maxRecipients;
        }
        size_t size() const
        {
            return to_.size() + cc_.size() + bcc_.size();
        }
        bool maxRecipientsSpecified() const
        {
            return maxRecipients != maxRecipientsUnspecified;
        }
        bool isLimitExeeded() const
        {
            return (maxRecipientsSpecified() && size() > maxRecipients);
        }

        EmailVec to() const {
            return to_;
        }
        EmailVec cc() const {
            return cc_;
        }

        EmailVec bcc() const {
            return bcc_;
        }

    private:
        bool findRecipient(const Email& email, EmailVec::iterator& result);

        bool replaceDisplayName(const Email& email);

        void fromString(const std::string & str, std::insert_iterator<EmailVec> insertIt);

        template<typename Out,typename F>
        void transformAll(Out ins, F func) const
        {
            transformVec(to_,ins,func);
            transformVec(cc_,ins,func);
            transformVec(bcc_,ins,func);
        }

        EmailVec to_;
        EmailVec cc_;
        EmailVec bcc_;
        size_t maxRecipients;
        std::set<std::string> addressesNotToBeLowercased_;
};

}

std::ostream & operator << ( std::ostream & s, const sendbernar::RecipientsRepository& repository );
std::string toString( const sendbernar::RecipientsRepository& repository );
