#include <mail/sendbernar/core/include/limits.h>
#include <mail/sendbernar/core/include/recipients_repository.h>
#include <boost/range/algorithm_ext/push_back.hpp>


namespace sendbernar {

yamail::expected<LimitsResponse> limits(
        const params::LimitsParams& p, const SmtpLimits& smtpLimits,
        std::size_t maxRecipients, std::size_t attachmentsMaxSize,
        std::size_t messageMaxSize, double limitsFactor) {

    LimitsResponse resp;

    if (p.recipients.to || p.recipients.cc || p.recipients.bcc) {
        resp.domains = LimitsResponse::Domains();
        RecipientsRepository recipients;

        recipients.setMaxRecipients(maxRecipients);

        try {
            if (p.recipients.to) {
                recipients.loadTo(*p.recipients.to);
            }

            if (p.recipients.cc) {
                recipients.loadCc(*p.recipients.cc);
            }

            if (p.recipients.bcc) {
                recipients.loadBcc(*p.recipients.bcc);
            }
        } catch(const std::exception& e) {
            return yamail::make_unexpected(make_error(ErrorResult::invalidParam, e.what()));
        }

        resp.domains->maxRecipientsReached = recipients.isLimitExeeded();

        std::vector<std::string> uniqDomains;
        boost::copy(recipients.toVector()
                    | boost::adaptors::transformed([](const Email& e) {
                        return e.domain();
                    })
                    , std::back_inserter(uniqDomains)
        );

        boost::push_back(resp.domains->limits
                         , boost::sort(uniqDomains)
                         | boost::adaptors::uniqued
                         | boost::adaptors::transformed([&](const std::string& domain) {
                            const auto l = smtpLimits.find(domain);
                            return LimitsResponse::DomainLimit{
                                domain,
                                l != smtpLimits.end() ? std::make_optional(l->second * limitsFactor)
                                                      : std::nullopt
                            };
                         })
        );
    }

    resp.config = LimitsResponse::Config {
        static_cast<std::size_t>(attachmentsMaxSize * limitsFactor),
        static_cast<std::size_t>(messageMaxSize * limitsFactor),
        static_cast<std::size_t>(maxRecipients),
    };

    return resp;
}

}
