#include <mail/sendbernar/core/include/metadata.h>
#include <macs_pg/integration/pa_profiler.h>
#include <macs_pg/service/service.h>
#include <macs_pg/service/factory.h>
#include <boost/range/adaptor/transformed.hpp>
#include <mail/sendbernar/core/include/config.h>

namespace sendbernar {

template<class Ctx>
struct MacsMetadata: public Metadata {
    macs::ServicePtr metadata_;
    Ctx ctx_;

    MacsMetadata(macs::ServicePtr metadata, Ctx ioResult)
        : metadata_(std::move(metadata))
        , ctx_(std::move(ioResult))
    { }

    macs::Envelope getById(const macs::Mid& mid) const override {
        return metadata_->envelopes().getById(mid, ctx_);
    }

    yamail::expected<macs::settings::ProfilePtr> getSettingsProfile(macs::settings::SettingsList settingsList) const override {
        macs::error_code ec;
        macs::settings::ProfilePtr profile = metadata_->settings().getProfile(std::move(settingsList), ctx_[ec]);
        if (ec) {
            return yamail::make_unexpected(ec);
        } else {
            return profile;
        }
    }

    std::vector<macs::Envelope> getEnvelopesWithNoAnswer(const std::string& messageId, const macs::Lid& lid) const override {
        const auto result = metadata_->envelopes().query()
                .withNoAnswer()
                .msgId(messageId)
                .includeLabel(lid)
                .get(ctx_);

        return std::vector<macs::Envelope>{std::begin(result), std::end(result)};
    }

    macs::MidVec getByMessageId(const std::string& messageId, const macs::FidVec& excludedFids) const override {
        auto ret = metadata_->envelopes().getByMessageIdWithExcluded(messageId, excludedFids, ctx_);

        return macs::MidVec(ret.begin(), ret.end());
    }

    std::string getMessageId(const macs::Mid& mid) const override {
        return metadata_->envelopes().getMessageId(mid, ctx_);
    }

    macs::Fid fidBySymbol(const macs::Folder::Symbol& symbol) const override {
        const macs::FolderSet folders = metadata_->folders().getAllFoldersWithHidden(ctx_);
        macs::Fid fid = folders.fid(symbol);
        if (symbol == macs::Folder::Symbol::template_ && fid.empty()) {
            return metadata_->folders().getOrCreateFolder(
                "template", folders.fid(macs::Folder::Symbol::drafts),
                macs::Folder::Symbol::template_, ctx_
            ).fid();
        }

        return fid;
    }

    std::pair<macs::error_code, macs::MidsWithMimes> getMimes(const macs::MidVec& mids) const override {
        macs::error_code ec;
        macs::MidsWithMimes mimes = metadata_->envelopes().getMimes(macs::Mids(mids.begin(), mids.end()),
                                                                    ctx_[ec]);
        return std::make_pair(ec, mimes);
    }

    mail_getter::MessageAccessWithWindatParams getMessageAccessWithWindatParams(const macs::Mid& mid) const override {
        return mail_getter::getMessageAccessWithWindatParams(metadata_->envelopes(), mid, ctx_);
    }

    mail_getter::MessageAccessParams getMessageAccessParams(const macs::Mid& mid) const override {
        return mail_getter::getMessageAccessParams(metadata_->envelopes(), mid, ctx_);
    }

    void unmarkEnvelopes(const macs::Mid& mids, const macs::Labels& labels) const override {
        this->metadata_->envelopes().unmarkEnvelopes({mids}, std::list<macs::Label>(labels.begin(), labels.end()),
                                                     this->ctx_);
    }

    void markEnvelopes(const macs::MidVec& mids, const macs::Labels& labels) const override {
        this->metadata_->envelopes().markEnvelopes(macs::Mids(mids.begin(), mids.end()),
                                                   std::list<macs::Label>(labels.begin(), labels.end()),
                                                   this->ctx_);
    }

    void deleteDraft(const macs::Mid& mid) const override {
        const macs::Envelope envelope = this->metadata_->envelopes().getById(mid,this->ctx_);

        this->metadata_->envelopes().forceRemove(envelope.mid(), this->ctx_);
        this->metadata_->envelopes().deleteFromStorage(envelope.stid(), this->ctx_);
    }

    void moveMessage(const macs::Fid& fid, const macs::Mid& mid) const override {
        this->metadata_->envelopes().moveMessage(fid, mid, this->ctx_);
    }

    macs::Label labelBySymbol(const macs::Label::Symbol& symbol) const override {
        return this->metadata_->labels().getOrCreateLabel(symbol, this->ctx_);
    }

    std::shared_ptr<Metadata> clone(boost::asio::yield_context yield) const override {
        const auto y = macs::io::make_yield_context(yield);
        return std::make_shared<MacsMetadata<decltype(y)>>(this->metadata_, y);
    }
};


ymod_maildb::ServiceParams serviceParams(params::CommonParams params) {
    return ymod_maildb::ServiceParams {
        .uid=std::move(params.uid),
        .realIp=std::move(params.realIp),
        .requestId=std::move(params.requestId),
        .module="sendbernar"
    };
}

MetadataPtr getMetadata(const params::CommonParams& params, const params::UserJournalParams& uj,
                        ContextLogger logger, ConfigPtr config) {
    macs::ServicePtr s = config->maildb->service(serviceParams(params), journalParams(uj),
                                                 getMacsPgLogger(logger), macs::pg::readNoLagReplicaThenMasterThenReplica);

    return std::make_shared<MacsMetadata<io_result::sync_context>>(
        std::move(s), macs::io::use_sync
    );
}

}
