#include <mail/sendbernar/core/include/recipients_repository.h>
#include <boost/range/algorithm/copy.hpp>
#include <boost/range/algorithm/find_if.hpp>
#include <butil/email/helpers.h>

namespace sendbernar {

bool RecipientsRepository::findRecipient(const Email& email, EmailVec::iterator& result)
{
    if ((result = EmailHelpers::findEqualEmail(to_, email)) != to_.end()) {
        return true;
    }
    if ((result = EmailHelpers::findEqualEmail(cc_, email)) != cc_.end()) {
        return true;
    }
    if ((result = EmailHelpers::findEqualEmail(bcc_, email)) != bcc_.end()) {
        return true;
    }
    return false;
}

bool RecipientsRepository::replaceDisplayName(const Email& email)
{
    EmailVec::iterator result;
    if (findRecipient(email,result)) {
        if (!email.displayName().empty() && result->displayName().empty()) {
            result->setDisplayName(email.displayName());
        }
        return true;
    }
    return false;
}

void RecipientsRepository::fromString( const std::string & str, std::insert_iterator<EmailVec> insertIt)
{
    for( rfc2822ns::address_iterator it( str ), end; it != end; ++it )
    {
        Email item(EmailHelpers::fromString(it));

        const auto domain = boost::find_if(addressesNotToBeLowercased_, [&item] (const auto& el) {
            return boost::iequals(el, item.domain());
        });
        if (domain != addressesNotToBeLowercased_.end()) {
            item = EmailHelpers::fromStringWithoutLowercase(it);
        }

        if (!replaceDisplayName(item)) {
            *insertIt =  item;
        }
    }
}

void RecipientsRepository::getDomains(std::set<std::string>& domains) const
{
    transformAll(std::inserter(domains,domains.end()),&Email::domain);
}

void RecipientsRepository::getAsVector(Recipients &recipients) const
{
    recipients.resize(to_.size() + cc_.size() + bcc_.size());
    std::copy(bcc_.begin(),bcc_.end(),
            std::copy(cc_.begin(),cc_.end(),
                std::copy(to_.begin(),to_.end(),recipients.begin())));
}

EmailVec RecipientsRepository::toVector() const {
    EmailVec ret;
    boost::copy(to_, std::back_inserter(ret));
    boost::copy(cc_, std::back_inserter(ret));
    boost::copy(bcc_, std::back_inserter(ret));
    return ret;
}

}

std::ostream & operator << ( std::ostream & s, const sendbernar::RecipientsRepository& repository ) {
    return s << toString( repository );
}

std::string toString( const sendbernar::RecipientsRepository& repository ) {
    return EmailHelpers::toString(repository.toVector());
}
