import os
import platform

import yaml

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import DogComponent
from mail.webmail_config.lib.make_config import make_config
from mail.pg.cachedb.devpack.cachedb import CacheDb


class BaseSendbernar(DogComponent):
    def __init__(self, env, components):
        super(BaseSendbernar, self).__init__(env, components, binary_name='sendbernar', custom_path='sendbernar')

    def _init_root(self):
        raise Exception('not implemented')

    def _generate_config(self, env):
        base = resource.find('sendbernar/base.yml')
        service_yaml = yaml.safe_load(resource.find('sendbernar/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        db_yaml = yaml.safe_load(resource.find('ymod_cachedb/service.yaml'))
        cfg = make_config(env, base, service_yaml, common_yaml, db_yaml, silent=True)
        config = self.yhttp.format_config(cfg)
        helpers.write2file(config, os.path.join(self.etc_path, 'sendbernar.yml'))

    def init_root(self):
        self.yhttp.init_root()

        helpers.write2file(resource.find('sendbernar/tvm_secret'),          os.path.join(self.etc_path, 'tvm_secret'))
        helpers.write2file(resource.find('sendbernar/no_answer.template'),  os.path.join(self.etc_path, 'no_answer.template'))
        helpers.write2file(resource.find('sendbernar/disk_attaches_title.template'),  os.path.join(self.etc_path, 'disk_attaches_title.template'))
        helpers.write2file(resource.find('sendbernar/barbet_restore_failed.template'), os.path.join(self.etc_path, 'barbet_restore_failed.template'))
        helpers.write2file(resource.find('sendbernar/barbet_create_failed.template'), os.path.join(self.etc_path, 'barbet_create_failed.template'))
        helpers.write2file(resource.find('sendbernar/smtplimits.lst'),      os.path.join(self.etc_path, 'smtplimits.lst'))
        helpers.write2file(resource.find('mail_getter/vdirect_keys'),       os.path.join(self.etc_path, 'vdirect_keys'))

        etc_cachedb_path = os.path.join(self.get_root(), 'etc', 'cachedb')
        helpers.mkdir_recursive(etc_cachedb_path)
        helpers.write2file(resource.find('ymod_cachedb/cachedb.conf'), os.path.join(etc_cachedb_path, 'cachedb.conf'))

        helpers.write2file(resource.find('content_type_detector_config/libmagic.issues'), os.path.join(self.etc_path, 'libmagic.issues'))
        helpers.write2file(resource.find('content_type_detector_config/magic'),           os.path.join(self.etc_path, 'magic'))
        helpers.write2file(resource.find('content_type_detector_config/mime.types'),      os.path.join(self.etc_path, 'mime.types'))
        helpers.write2file(resource.find('content_type_detector_config/mime-types.xml'),  os.path.join(self.etc_path, 'mime-types.xml'))

        etc_macs_pg_path = os.path.join(self.get_root(), 'etc', 'macs_pg')
        helpers.mkdir_recursive(etc_macs_pg_path)
        helpers.write2file(resource.find('macs_pg/query.conf'), os.path.join(etc_macs_pg_path, 'query.conf'))

        etc_recognizer_path = os.path.join(self.get_root(), 'etc', 'recognizer')
        helpers.mkdir_recursive(etc_recognizer_path)
        helpers.write2file(resource.find('recognizer/dict.dict'),        os.path.join(etc_recognizer_path, 'dict.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.dict'),    os.path.join(etc_recognizer_path, 'queryrec.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.weights'), os.path.join(etc_recognizer_path, 'queryrec.weights'))

        helpers.write2file(resource.find('attach_sid_aes_keys.json'), os.path.join(self.etc_path, 'attach_sid_aes_keys.json'))
        helpers.write2file(resource.find('attach_sid_hmac_keys.json'), os.path.join(self.etc_path, 'attach_sid_hmac_keys.json'))

        self.yhttp.init_pgpass(self.get_root())

        self._init_root()

    def start(self):
        self.yhttp.put_pgpassfile_in_env(self.get_root())
        self.yhttp.start('pong')


class SendbernarTesting(BaseSendbernar):
    NAME = 'sendbernar_testing'
    DEPS = []

    def __init__(self, env, components):
        super(SendbernarTesting, self).__init__(env, components)

    def _init_root(self):
        replace = {
            'callback_host': platform.node()+":"+str(self.webserver_port())
        }

        devpack = self.yhttp.format_config(resource.find('sendbernar/devpack.yml'), **replace)

        helpers.write2file(devpack, os.path.join(self.etc_path, 'sendbernar-devpack.yml'))
        self._generate_config('testing')

        with open('sendbernar.properties-testing', 'w') as f:
            f.write('''
sendbernar.host=http://localhost
sendbernar.port={sendbernar_port}
sendbernar.root_path={root}
                '''.format(sendbernar_port=self.webserver_port(), root=os.path.join(self.etc_path, os.pardir, os.pardir)))


class SendbernarDevelopment(BaseSendbernar):
    NAME = 'sendbernar_development'
    DEPS = []

    def __init__(self, env, components):
        super(SendbernarDevelopment, self).__init__(env, components)

    def _init_root(self):
        replace = {
            'callback_host': platform.node()+":"+str(self.webserver_port())
        }

        self._generate_config('development')
        devpack = self.yhttp.format_config(resource.find('sendbernar/devpack.yml'), **replace)

        helpers.write2file(devpack, os.path.join(self.etc_path, 'sendbernar-devpack.yml'))

        with open('sendbernar.properties-development', 'w') as f:
            f.write('''
sendbernar.host=http://localhost
sendbernar.port={sendbernar_port}
sendbernar.root_path={root}
                '''.format(sendbernar_port=self.webserver_port(), root=os.path.join(self.etc_path, os.pardir, os.pardir)))


class SendbernarDevelopmentWithCacheDB(BaseSendbernar):
    NAME = 'sendbernar_development_with_cachedb'
    DEPS = [CacheDb]

    def __init__(self, env, components):
        super(SendbernarDevelopmentWithCacheDB, self).__init__(env, components)

    def _init_root(self):
        replace = {
            'callback_host': platform.node()+":"+str(self.webserver_port()),
            'connection_string': f'host=localhost port={self.components[CacheDb].port()} user=sendbernar dbname=cachedb'
        }

        self._generate_config('development')
        devpack = self.yhttp.format_config(resource.find('sendbernar/with_cachedb.yml'), **replace)

        helpers.write2file(devpack, os.path.join(self.etc_path, 'sendbernar-devpack.yml'))

        with open('sendbernar.properties-development-with-cachedb', 'w') as f:
            f.write('''
sendbernar.host=http://localhost
sendbernar.port={sendbernar_port}
sendbernar.root_path={root}
                '''.format(sendbernar_port=self.webserver_port(), root=os.path.join(self.etc_path, os.pardir, os.pardir)))
