#!/usr/bin/env python3.5
# encoding: utf-8

import os
import yaml
import json
import requests
import shutil
import filecmp
import subprocess
import argparse
import datetime
from random import randint
from time import sleep


def get_local_dc():
    with open(meta_file, 'r') as f:
        local_meta = json.load(f)
        #print(local_meta)
        dc = local_meta['datacenter'].lower()
        #print(dc)
    return dc


def get_local_ipv6():
    with open(meta_file, 'r') as f:
        local_meta = json.load(f)
        #print(local_meta)
        ipv6 = local_meta['ipv6'].lower().split('/')[0]
        #print(ipv6)
    return ipv6


def get_api_meta():
    try:
        res = requests.get(api_endpoint, {'format': 'json', 'state': '*', 'component': '*'})
        #print(res.content)
        #print(res.status_code)
    except requests.exceptions.RequestException as e:
        print('[ERROR] Something went wrong:')
        print(e)
        exit(1)

    try:
        meta = json.loads(res.content.decode('utf-8'))
        for c in components:
            name = c['name']
            #print(meta['components'][name])
    except json.JSONDecodeError as e:
        print('[ERROR] Something went wrong:')
        print(e)
        exit(1)

    return meta


def create_instances_dict():
    component_instances = {}

    for c in components:
        name = c['name']
        component_instances[name] = []
        for inst in api_meta['components'][name]:
            #print(inst)

            if local_dc in inst['fqdn'].split('-')[0]:
                component_instances[name].append(inst)

    for c in components:
        name = c['name']
        #print(component_instances[name])

    return component_instances


def create_config_files():
    for c in components:
        #print(c)

        name = c['name']

        local_port = None
        if 'local_port' in c:
            local_port = c['local_port']

        remote_port = c['remote_port']

        remote_threads = None
        if 'remote_threads' in c:
            remote_threads = c['remote_threads']

        config_file = config_dir + '/service.include.new'

        try:
            with open(config_file, 'w') as f:
                # create upstream for local hound
                if local_port is not None:
                    f.write('upstream ' + name + '-backend {\n    least_conn;\n')
                    f.write('    server [::1]:' + str(local_port) + ';\n')
                    f.write('}\n\n')

                # create upstream for remote hounds
                f.write('upstream ' + name + '-backend-fallback {\n    least_conn;\n')

                for inst in component_instances[name]:
                    ip = inst['ip']

                    port = remote_port
                    if remote_threads is not None:
                        while port < remote_port + remote_threads:
                            f.write('    server [' + ip + ']:' + str(port) + ';\n')
                            port += 1
                    else:
                        f.write('    server [' + ip + ']:' + str(port) + ';\n')

                f.write('}\n')

            print('[INFO] Successfully created ' + config_file)
        except Exception as e:
            print('[ERROR] Something went wrong:')
            print(e)
            exit(1)
    return


def compare_files(src_postfix, dst_postfix):
    for c in components:
        name = c['name']
        config_basename = config_dir + '/service'
        src_config = config_basename + src_postfix
        dst_config = config_basename + dst_postfix

        eq = filecmp.cmp(src_config, dst_config, shallow=False)
        #print('Configs equal: ' + str(eq))
        if eq is False:
            return False
        else:
            continue
    return True


def copy_configs(src_postfix, dst_postfix):
    for c in components:
        name = c['name']
        config_basename = config_dir + '/service'
        src_config = config_basename + src_postfix
        dst_config = config_basename + dst_postfix

        try:
            shutil.copyfile(src_config, dst_config)
            print('[INFO] Copied ' + src_postfix + ' to ' + dst_postfix + ' for ' + str(name))
        except shutil.Error as e:
            print('[ERROR] Something went wrong:')
            print(e)
            exit(1)
    return


def reload_nginx():
    cmd = 'nginx -t'
    p = subprocess.Popen(cmd.split(), stdout=open('/dev/null'), stderr=subprocess.PIPE)
    status_line = p.communicate()[1].splitlines()[-1]

    if 'test is successful' in str(status_line):
        print('[INFO] Nginx config test successful')
    else:
        print('[ERROR] Nginx config test failed')
        copy_configs('.include.bak', '.include')
        exit(1)

    if do_reload_nginx == True:
        cmd = 'nginx -s reload'
        print('[INFO] Some random sleep')
        sleep(randint(1,60))
        p = subprocess.Popen(cmd.split(), stdout=open('/dev/null'), stderr=open('/dev/null'))
        print('[INFO] Nginx reloaded')

    return


if __name__ == '__main__':

    parser = argparse.ArgumentParser(description='nginx upstream config generator')
    parser.add_argument('-r', help='reload nginx', action='store_true')

    args = parser.parse_args()

    if args.r:
        do_reload_nginx = True
    else:
        do_reload_nginx = False

    ts = datetime.datetime.now()
    print(str(ts))
    print('[INFO] Started')

    config_file = os.path.dirname(os.path.realpath(__file__)) + '/nginx_upstream_gen_config.yml'

    with open(config_file, 'r') as f:
        config = yaml.load(f)

    meta_file = config['meta_file']
    api_endpoint = config['api_endpoint']
    config_dir = config['config_dir']
    components = config['components']

    local_dc = get_local_dc()
    local_ipv6 = get_local_ipv6()
    api_meta = get_api_meta()
    component_instances = create_instances_dict()
    create_config_files()
    configs_equal = compare_files('.include', '.include.new')

    if configs_equal is False:
        print('[INFO] Found changes in config files')
        copy_configs('.include', '.include.bak')
        copy_configs('.include.new', '.include')
        reload_nginx()
    else:
        print('[INFO] No changes in config files')

    print('[INFO] All done')
