import argparse
import yaml
import requests
import subprocess
import os


SENDBERNAR_CONFIG = '/etc/sendbernar/sendbernar.yml'
TVM_SOURCE_PATH = '/tvm2_root_client_id'
CALLMEBACK_CONFIG_PATH = 'configuration.send.reminders.client_configuration.add.client.nodes'


def get_service_ticket(login, srcId, dstId):
    tvmknife_call = f'tvmknife get_service_ticket sshkey -s {srcId} -d {dstId} -l {login}'
    process = subprocess.Popen(tvmknife_call.split(), stdout=subprocess.PIPE, stderr=None, text=True)
    output = process.communicate()[0].rstrip()
    return output


def get_ssh_user():
    ssh_user = os.environ.get('SSH_USER')
    if ssh_user == 'root':
        ssh_add_call = 'ssh-add -l'
        process = subprocess.Popen(ssh_add_call.split(), stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
        output, error = process.communicate()
        if error:
            raise Exception(f'Error while getting SSH_USER by ssh-add: {error}')
        address = output.split()[2]
        ssh_user = address.split('@')[0]
    return ssh_user


def get_tvm_id_from_config(config_path):
    f = open(config_path)
    return f.readline()


def get_module_config(data, name):
    return next(v for v in data['config']['modules']['module'] if v['system']['name'] == name)


def get_callmebacks_tvm_id_from_config(config):
    yaml_config = yaml.safe_load(config)
    mod = get_module_config(yaml_config, 'ymod_tvm')
    return next(filter(lambda m: m['name'] == 'reminders_api', mod['configuration']['target_services']))['id']


def get_config_value_by_path(config_map, path):
    current_map = dict(config_map)
    for key in path.split('.'):
        if not key:
            raise Exception('Key in the callmeback_cfg_path should not be empty')
        elif key not in current_map:
            raise Exception(f'Wrong key in the callmeback_cfg_path: {key}')
        else:
            current_map = current_map[key]
    return current_map


def get_callmebacks_url_from_config(config, config_path, service):
    yaml_config = yaml.safe_load(config)
    mod = get_module_config(yaml_config, service)
    url = get_config_value_by_path(mod, config_path)
    if type(url) is not str:
        raise Exception(f'Wrong callmeback_cfg_path to the url, returns an object of {type(url)}')
    return url


def make_request(dst_url, action, uid, reminder_id, service_ticket):
    url = None
    resp = None
    headers={'X-Ya-Service-Ticket': service_ticket}
    if action == 'get':
        url = f'{dst_url}/v1/event/{uid}/{reminder_id}'
        url = url.rstrip('/')
        resp = requests.get(url, headers=headers)
    else:
        url = f'{dst_url}/v1/event/{action}/{uid}/{reminder_id}'
        url = url.rstrip('/')
        resp = requests.post(url, headers=headers)
    print(f'generated url : {url}')
    return resp.text


def call(service, service_config, callmeback_config_path, tvm_source_path):
    parser = argparse.ArgumentParser(description=f'Callmeback caller of /v1/event/action/uid/reminder_id url from {service}', add_help=True)
    optional_args = parser.add_argument_group('optional arguments')
    optional_args.add_argument('--reminder_id', dest='reminder_id', help='A part of the calling url', default='')
    optional_args.add_argument('--config', dest='config_path', help='A path to the service config, %(default)s by default', default=service_config)
    optional_args.add_argument('--callmeback_cfg_path', dest='callmeback_cfg_path',
                                help='Config path to find callmeback url in the config.modules.module, %(default)s by default',
                                default=callmeback_config_path)
    tvm_id_args = optional_args.add_mutually_exclusive_group()
    tvm_id_args.add_argument('--tvm_file', dest='tvm_file_path', help='A path to the root client id, %(default)s by default', default=tvm_source_path)
    tvm_id_args.add_argument('--tvm_id', dest='tvm_id', help='f{service} tvm id')
    required_args = parser.add_argument_group('required arguments')
    required_args.add_argument('--action', dest='action', help='A part of the calling url', required=True)
    required_args.add_argument('--uid', dest='uid', help='A part of the calling url', required=True)

    args = parser.parse_args()

    src_id = args.tvm_id if args.tvm_id else get_tvm_id_from_config(args.tvm_file_path)
    dst_id = get_callmebacks_tvm_id_from_config(open(args.config_path))
    login = get_ssh_user()
    service_ticket = get_service_ticket(login, src_id, dst_id)
    dst_url = get_callmebacks_url_from_config(open(args.config_path), args.callmeback_cfg_path, service)
    print(make_request(dst_url, args.action, args.uid, args.reminder_id, service_ticket))


def main():
    call('sendbernar', SENDBERNAR_CONFIG, CALLMEBACK_CONFIG_PATH, TVM_SOURCE_PATH)
