#include <mail/sendbernar/server/include/callbacks/send_delayed_message.h>
#include <mail/sendbernar/server/include/fwd.h>
#include <mail/sendbernar/services/include/smtp_gate.h>
#include <mail/sendbernar/services/include/delayed.h>
#include <mail/sendbernar/composer/include/delayed_message_compose.h>
#include <mail/sendbernar/client/include/category.h>
#include <mail/sendbernar/services/include/postprocess.h>
#include <mail/sendbernar/composer/include/delivery_helpers.h>
#include <boost/variant.hpp>


namespace sendbernar::server::callbacks {

inline bool stopRetrying(DeliveryResult result) {
    return isSpamOrVirus(result) || !needToSaveAsDraft(result);
}

void DelayedMessage::doSendbernar(Context& ctx, const Account& a,
                                  params::CommonParams common, MetadataPtr m) const {
    const auto p = getContinueSendingMessage(ctx);

    DelayedMessageComposer cmp(common, ctx.logger);

    auto nts = cmp.needToSend(p.mid, m);

    if (boost::get<DelayedMessageCallbackResponse>(&nts)) {
        ctx.responseWith(DelayedMessageCallbackResponse());
    } else if (auto* ec = boost::get<mail_errors::error_code>(&nts)) {
        ctx.responseWith(*ec);
    } else {
        compose::SendMessage msg = cmp.message(a, p.mid, m, sendConfig());

        const RecipientsRepository& recipients = msg.recipients;

        if (!checkMentionsInRecipients(p.mentions, recipients)) {
            ctx.responseWith(make_error(ErrorResult::invalidParam, "there are mentioned emails missing in recipients"));
            return;
        }

        const NwSmtp smtp(common, ctx.metrics, ctx.http, sendConfig()->nwsmtp());

        SourceMid sourceMid(p.mid, m, ctx.logger);
        SendResult result;
        if (sourceMid.isUndo()) {
            result = smtp.sendDelayed(p.notify_on_send, cmp.lids(), p.mid, p.mentions, msg);
        } else {
            result = smtp.send(p.notify_on_send, cmp.lids(), p.mid, p.mentions,msg);
        }

        if (result.deliveryResult == DeliveryResult::ok) {
            ctx.responseWith(DelayedMessageCallbackResponse());

            PostProcessMessage obj(ctx.httpPtr, std::move(common), getUserJournalParams(ctx),
                                   p.postprocess, config_, std::move(a), m, ctx.logger);

            spawnCoroutine([o = std::move(obj)] (boost::asio::yield_context yield) mutable {
                o.make(yield);
            });
        } else if (stopRetrying(result.deliveryResult)) {
            const macs::Label sendingFailed = m->labelBySymbol(macs::Label::Symbol::sending_failed_label);
            m->markEnvelopes({ p.mid }, { sendingFailed });

            LOGDOG_(ctx.logger, notice, log::message="callback delayed_message stop sending");

            ctx.responseWith(CallbackResponseError{true});
        } else {
            ctx.responseWith(CallbackResponseError{false});
        }
    }
}

}

SBR_HANDLER_IMPL(sendbernar::server::callbacks, DelayedMessage)
