#include <mail/sendbernar/server/include/context.h>

namespace sendbernar {
namespace server {

boost::optional<std::string> getOptionalHeader(const ymod_webserver::request_ptr& request, const std::string& name) {
    const std::string n = boost::algorithm::to_lower_copy(name);
    auto iter = request->headers.find(n);

    return iter == request->headers.end() ? boost::none : boost::make_optional(iter->second);
}

Context::Context(const ymod_webserver::http::stream_ptr& stream, bool logResponse, const http_getter::TypedClientModule& httpModule)
    : request(stream->request())
    , metrics(getRequestMetrics(*this, stream->request()->url.make_full_path()))
    , logger(getContextLogger(requestId(),
                              optionalArg("uid")))
    , method(stream->request()->url.make_full_path())
    , response(stream)
    , httpPtr(
        httpModule.create(
            *stream->request(),
            withLogAndUnistat(
                httpModule.httpLogger(
                    optionalArg("uid").get_value_or(""),
                    requestId()
                ),
                metrics
            )
        )
    )
    , http(*httpPtr)
    , logResponse(logResponse)
{ }

boost::optional<std::vector<std::string>> Context::optionalArgs(const std::string& name) const {
    const auto range = request->url.params.equal_range(name);
    if (range.first == request->url.params.end()) {
        return boost::none;
    }
    std::vector<std::string> ret;
    boost::copy(range | boost::adaptors::map_values, std::back_inserter(ret));

    return ret;
}

boost::optional<std::string> Context::optionalArg(const std::string& name) const {
    auto iter = request->url.params.find(name);
    return iter == request->url.params.end() ? boost::none : boost::make_optional(iter->second);
}

boost::optional<std::string> Context::optionalHeader(const std::string& name) const {
    return server::getOptionalHeader(request, name);
}

std::string Context::rawBody() const {
    return std::string(request->raw_body.begin(), request->raw_body.end());
}

std::string Context::requestId() const {
    return optionalHeader("X-Request-Id").get_value_or("");
}

void Context::responseWith(response::Response resp) {
    if (resp.errorCode) {
        LOGDOG_(logger, error, log::where_name=method, log::error_code=resp.errorCode);
        metrics.mailSendAndSendbernarErrorCode(resp.errorCode);
    }

    response->set_code(resp.code);
    response->set_content_type(resp.contentType.first, resp.contentType.second);
    for (const auto& p: resp.additionalHeaders.flatten()) {
        response->add_header(p.first, p.second);
    }

    response->result_body(resp.body);
}

}
}
