#include <mail/sendbernar/server/include/handlers/send_delayed.h>
#include <mail/sendbernar/server/include/fwd.h>
#include <mail/sendbernar/services/include/smtp_gate.h>
#include <mail/sendbernar/composer/include/mail_compose.h>
#include <mail/sendbernar/composer/include/delivery_helpers.h>
#include <mail/sendbernar/services/include/delayed.h>
#include <mail/sendbernar/services/include/reminders.h>
#include <mail/sendbernar/services/include/captcha.h>
#include <mail/sendbernar/core/include/sharpei.h>
#include <mail/sendbernar/services/include/make_responses.h>
#include <mail/sendbernar/services/include/postprocess.h>
#include <mail/sendbernar/server/include/compose.h>


namespace sendbernar::server::handlers {

void SendDelayedMessage::doSendbernar(Context& ctx, const Account& a,
                                      params::CommonParams common, MetadataPtr m) const {

    const auto p = params(ctx);
    const auto cfg = sendConfig();
    const auto& http = ctx.http;

    if (!isMasterAlive(common.uid, cfg->sharpei(), http)) {
        ctx.responseWith(make_error(ErrorResult::cannotSaveMessage, "cannot save message, master is dead"));
        return;
    }

    SourceMid sourceMid(p.send.message, m, ctx.logger);
    auto deleteCode = deleteDelayedReminderSync(common.uid, sourceMid, http, cfg);
    if (deleteCode) {
        ctx.responseWith(deleteCode);
        return;
    }

    const auto captchaStatus = checkCaptcha(p.send.captcha, cfg->captcha(), http, p.send.captcha_passed);
    compose::Expected<compose::SendDelayed> composed = mailCompose(cfg, ctx, a, m, sourceMid, common, p, captchaStatus);

    if (!composed) {
        ctx.responseWith(make_error(composed.error().result), composed.error().messageId);
        return;
    }

    compose::SendDelayed& msg = composed.value();
    const RecipientsRepository& recipients = msg.recipients;

    if (!checkMentionsInRecipients(p.send.mentions, recipients)) {
        ctx.responseWith(make_error(ErrorResult::invalidParam, "there are mentioned emails missing in recipients"));
        return;
    }

    MessageSavedResponse resp = makeSaveResponse(msg, m->fidBySymbol(macs::Folder::Symbol::outbox),
                                                 cfg->mimeTypesList());
    DeliveryResult result;

    auto lids = filteredLids(p.send, m);
    const SendConfiguration::NwSmtp& sendCfg = cfg->nwsmtp();

    const NwSmtp nwSmtp(common, ctx.metrics, http, sendCfg);
    std::tie(result, resp.stored.mid) = nwSmtp.saveDelayed(symbolsToSet_, lids, resp.stored.fid,
                                                             sourceMid.optional(), msg);

    if (result == DeliveryResult::ok) {
        params::PostProcessMessage postParams = makePostprocessParams(msg, lids,
                                                                      boost::make_optional(resp.stored.mid), p.send);

        params::UserJournalParams uj = getUserJournalParams(ctx);
        params::ContinueSendingMessage contParams = {
            resp.stored.mid,
            p.send.delivery.confirm_delivery.get_value_or(false),
            p.send.mentions,
            std::move(postParams)
        };

        mail_errors::error_code ec = createDelayedReminder(common, uj, p, contParams, sourceMid, http,
                                                           cfg, ctx.logger);
        if (ec) {
            ctx.responseWith(response::SaveMessageWithOptionalCaptcha {
                .result=result,
                .response=std::move(resp)
            });
        } else {
            ctx.responseWith(std::move(resp));
        }
    } else if (result == DeliveryResult::spam && p.send.captcha_passed.value_or(false)) {
        ctx.responseWith(make_error(ErrorResult::unexpectedCondition, "passed captcha for send_delayed but still got marked as spam"));
    } else if (!needToSaveAsDraft(result)) {
        ctx.responseWith(make_error(result));
    } else {
        CaptchaResult captcha;
        if (checkCaptcha(result)) {
            if (auto c = Captcha(cfg->captcha()).request(a.karmaValue, p.send.captcha_type, http)) {
                captcha = *c;
            } else {
                ctx.responseWith(make_error(ErrorResult::unexpectedCondition, msg.messageId));
                return;
            }
        }

        const macs::Fid drafts = m->fidBySymbol(macs::Folder::Symbol::drafts);
        auto saved = makeSaveResponse(msg, drafts, cfg->mimeTypesList());

        DeliveryResult saveResult;
        std::vector<macs::Label::Symbol> symbols = { macs::Label::Symbol::seen_label, macs::Label::Symbol::draft_label };
        std::tie(saveResult, saved.stored.mid) = nwSmtp.save(symbols, lids, saved.stored.fid,
                                                               sourceMid.optional(), msg);

        const bool savingFailed = saveResult != DeliveryResult::ok;
        if (savingFailed) {
            ctx.responseWith(make_error(saveResult));
        } else {
            ctx.responseWith(response::SaveMessageWithOptionalCaptcha {
                .result=result,
                .captcha=captcha,
                .response=saved
            });
        }
    }
}

}

SBR_HANDLER_IMPL(sendbernar::server::handlers, SendDelayed)
SBR_HANDLER_IMPL(sendbernar::server::handlers, SendUndo)
