#include <mail/sendbernar/server/include/handlers/send_message.h>
#include <mail/sendbernar/server/include/fwd.h>
#include <mail/sendbernar/services/include/smtp_gate.h>
#include <mail/sendbernar/composer/include/mail_compose.h>
#include <mail/sendbernar/composer/include/delivery_helpers.h>
#include <mail/sendbernar/services/include/delayed.h>
#include <mail/sendbernar/services/include/captcha.h>
#include <mail/sendbernar/core/include/sharpei.h>
#include <mail/sendbernar/services/include/postprocess.h>
#include <mail/sendbernar/services/include/make_responses.h>
#include <mail/sendbernar/services/include/operation_id.h>
#include <mail/sendbernar/server/include/compose.h>


namespace sendbernar::server::handlers {

void SendMessage::doSendbernar(Context& ctx, const Account& a,
                               params::CommonParams common, MetadataPtr m) const {

    const auto p = getSendMessage(ctx);
    auto cfg = sendConfig();
    const auto& http = ctx.http;


    SourceMid sourceMid(p.message, m, ctx.logger);
    deleteDelayedReminder(common.uid, sourceMid, http, cfg);

    const auto captchaStatus = checkCaptcha(p.captcha, cfg->captcha(), http, p.captcha_passed);

    if (checkAndResponseOperationId(p.operation_id, common.uid, *config_->cachedb, ctx)) {
        return;
    }

    compose::Expected<compose::SendMessage> composed = mailCompose(cfg, ctx, a, m, sourceMid, common, p, captchaStatus);

    if (!composed) {
        ctx.responseWith(make_error(composed.error().result), composed.error().messageId);
        return;
    }
    compose::SendMessage& msg = composed.value();

    const RecipientsRepository& recipients = msg.recipients;

    if (!checkMentionsInRecipients(p.mentions, recipients)) {
        ctx.responseWith(make_error(ErrorResult::invalidParam, "there are mentioned emails missing in recipients"));
        return;
    }

    const std::vector<macs::Lid> lids = filteredLids(p, m);
    const NwSmtp smtp(common, ctx.metrics, http, cfg->nwsmtp());

    SendResult sendResult = smtp.send(p.delivery.confirm_delivery.get_value_or(false),
                            lids, sourceMid.optional(), p.mentions, msg);


    if (sendResult.deliveryResult == DeliveryResult::ok) {
        saveOperationId(p.operation_id, common.uid, msg.messageId, *config_->cachedb, ctx);
        ctx.responseWith(makeSentResponse(msg));

        auto postParams = makePostprocessParams(msg, lids, sourceMid, p);
        PostProcessMessage obj(ctx.httpPtr, std::move(common), getUserJournalParams(ctx),
                               std::move(postParams), config_, std::move(a), m, ctx.logger);
        spawnCoroutine([o = std::move(obj)] (boost::asio::yield_context yield) mutable {
            o.make(yield);
        });
    } else if (sendResult.deliveryResult == DeliveryResult::spam && p.captcha_passed.value_or(false)) {
        ctx.responseWith(make_error(ErrorResult::unexpectedCondition, "passed captcha for send_message but still got marked as spam"));
    } else if (!needToSaveAsDraft(sendResult.deliveryResult)) {
        ctx.responseWith(make_error(sendResult.deliveryResult));
    } else {
        if (!isMasterAlive(common.uid, cfg->sharpei(), http)) {
            ctx.responseWith(make_error(ErrorResult::cannotSaveMessage, "cannot save message, master is dead"));
            return;
        }

        CaptchaResult captcha;
        if (checkCaptcha(sendResult.deliveryResult)) {
            if (auto c = Captcha(cfg->captcha()).request(a.karmaValue, p.captcha_type, http)) {
                captcha = *c;
            } else {
                ctx.responseWith(make_error(ErrorResult::unexpectedCondition, msg.messageId));
                return;
            }
        }

        const macs::Fid drafts = m->fidBySymbol(macs::Folder::Symbol::drafts);
        auto saved = makeSaveResponse(msg, drafts, cfg->mimeTypesList());

        DeliveryResult saveResult;
        std::vector<macs::Label::Symbol> symbols = { macs::Label::Symbol::seen_label, macs::Label::Symbol::draft_label };
        const NwSmtp nwSmtp(common, ctx.metrics, http, cfg->nwsmtp());
        std::tie(saveResult, saved.stored.mid) = nwSmtp.save(symbols, lids, saved.stored.fid,
                                                 sourceMid.optional(), msg);

        const bool savingFailed = saveResult != DeliveryResult::ok;
        if (savingFailed) {
            ctx.responseWith(make_error(saveResult));
        } else {
            ctx.responseWith(response::SaveMessageWithOptionalCaptcha {
                .result=sendResult.deliveryResult,
                .captcha=captcha,
                .response=saved,
                .banReason=sendResult.banReason
            });
        }
    }
}

}

SBR_HANDLER_IMPL(sendbernar::server::handlers, SendMessage)
SBR_HANDLER_IMPL(sendbernar::server::handlers, SendService)
