#include <mail/sendbernar/server/include/remind_message_sender.h>
#include <mail/sendbernar/services/include/smtp_gate.h>
#include <mail/sendbernar/services/include/msg_field_builder.h>
#include <tuple>


namespace sendbernar {
namespace server {

std::string msgBody(const RemindMsgFieldBuilder& msgFieldBuilder,
                    SendConfigurationPtr config, const macs::Envelope& envelope,
                    const std::string& originalDate) {
    const std::string link = config->wmiServer() + "/message?ids=" + envelope.mid();

    std::vector<std::string> recipients;

    if (!envelope.to().empty()) {
        recipients.push_back("To: " + envelope.to());
    }

    if (!envelope.cc().empty()) {
        recipients.push_back("Cc: " + envelope.cc());
    }

    return msgFieldBuilder.buildBody(envelope.subject(),
                                     originalDate, boost::algorithm::join(recipients, ", "),
                                     link, envelope.from());
}

const RemindTemplate* findReminderTemplate(const std::string& lang,
                                           const RemindTemplates& templ) {
    static const std::string DEF_REMIND_TEMPLATE_LANG = "ru";
    auto it = templ.find(lang);
    if (it == templ.end()) {
        it = templ.find(DEF_REMIND_TEMPLATE_LANG);
        if (it == templ.end()) {
            return 0;
        }
    }
    return &(it->second);
}

mail_errors::error_code RemindMessageSender::composeAndSend(const std::string& uniqId) {
    const RemindTemplate* remind = findReminderTemplate(lang_, templates_);
    if (!remind) {
        return make_error(ErrorResult::reminderError, "remind templates not initialized");
    }

    const RemindMsgFieldBuilder msgFieldBuilder(*remind);

    SystemMailComposer composer = SystemMailComposer().desiredCharset("utf-8")
            .fromMailbox(msgFieldBuilder.buildFromMailbox())
            .fromName(msgFieldBuilder.buildFromName())
            .to(to_)
            .messageId(messageId_)
            .subject(msgFieldBuilder.subject())
            .subtype("html")
            .body(msgFieldBuilder.buildBody(uniqId))
            .inReplyTo(inReplyTo_)
            .messageMaxSize(config_->messageMaxSize());

    const RecipientsRepository& recipients = composer.recipients();

    auto composed = composer.compose();

    if (!composed) {
        return make_error(composed.error());
    }

    NwSmtp nw(common_, ctx_.metrics, ctx_.http, config_->nwsmtp());
    DeliveryResult nwResult = nw.sendSystemMessage({"SystMetkaSO:people",
                                                    "SystMetkaSO:reminder",
                                                    "SystMetkaSO:greeting"},
                                                   { }, { }, recipients, std::move(composed.value())).deliveryResult;

    if (nwResult == DeliveryResult::ok) {
        return mail_errors::error_code();
    } else {
        return make_error(nwResult);
    }
}

mail_errors::error_code RemindMessageSender::composeAndSend(const macs::Label& label,
                                                            const macs::Envelope& envelope) {
    const RemindTemplate* remind = findReminderTemplate(lang_, templates_);
    if (!remind) {
        return make_error(ErrorResult::reminderError, "remind templates not initialized");
    }

    const RemindMsgFieldBuilder msgFieldBuilder(*remind);

    SystemMailComposer composer = SystemMailComposer().desiredCharset("utf-8")
            .fromMailbox(msgFieldBuilder.buildFromMailbox())
            .fromName(msgFieldBuilder.buildFromName())
            .to(to_)
            .messageId(messageId_)
            .subject(msgFieldBuilder.buildSubject(envelope.subject()))
            .subtype("html")
            .body(msgBody(msgFieldBuilder, config_, envelope, date_))
            .inReplyTo(inReplyTo_)
            .messageMaxSize(config_->messageMaxSize());

    const RecipientsRepository& recipients = composer.recipients();

    auto composed = composer.compose();

    if (!composed) {
        return make_error(composed.error());
    }

    NwSmtp nw(common_, ctx_.metrics, ctx_.http, config_->nwsmtp());
    DeliveryResult nwResult = nw.sendSystemMessage({"SystMetkaSO:people",
                                                    "SystMetkaSO:reminder",
                                                    "SystMetkaSO:greeting"},
                                                   { }, { label.lid() }, recipients, std::move(composed.value())).deliveryResult;

    if (nwResult == DeliveryResult::ok) {
        return mail_errors::error_code();
    } else {
        return make_error(nwResult);
    }
}

}
}
