#include <mail/sendbernar/services/include/blackbox.h>
#include <yamail/data/deserialization/json_reader.h>
#include <mail/sendbernar/client/include/error_result.h>
#include <butil/network/utils.h>
#include <butil/network/idn.h>


#define AS_IS(TYPE, NAME)                       (NAME, TYPE, TYPE, obj.NAME, obj.NAME = val)
#define CHANGE_NAME(TYPE, CONF_NAME, NAME)      (CONF_NAME, TYPE, TYPE, obj.NAME, obj.NAME = val)

BOOST_FUSION_DEFINE_STRUCT((sendbernar)(blackbox), Uid,
    (std::string, value)
)

BOOST_FUSION_DEFINE_STRUCT((sendbernar)(blackbox), KarmaValue,
    (std::string, value)
)

BOOST_FUSION_DEFINE_STRUCT((sendbernar)(blackbox), KarmaStatus,
    (std::string, value)
)

namespace sendbernar {
namespace blackbox {
struct Address {
    std::string address;
    bool default_ = false;
    bool native = false;
    bool validated = false;
};

struct DbFields {
    std::string country;
    std::string tz;
    std::string login;
    std::string suid;
    std::string lang;
    std::string firstname;
    std::string lastname;
};
}
}

YREFLECTION_ADAPT_ADT(sendbernar::blackbox::Address,
    AS_IS(std::string, address)
    CHANGE_NAME(bool, default, default_)
    AS_IS(bool, native)
    AS_IS(bool, validated)
)

YREFLECTION_ADAPT_ADT(sendbernar::blackbox::DbFields,
    CHANGE_NAME(std::string, account_info.country.uid, country)
    CHANGE_NAME(std::string, account_info.tz.uid, tz)
    CHANGE_NAME(std::string, subscription.login.2, login)
    CHANGE_NAME(std::string, subscription.suid.2, suid)
    CHANGE_NAME(std::string, userinfo.lang.uid, lang)
    CHANGE_NAME(std::string, userinfo.firstname.uid, firstname)
    CHANGE_NAME(std::string, userinfo.lastname.uid, lastname)
)

namespace sendbernar {
namespace blackbox {

using Aliases = std::map<std::string, std::string>;

struct User {
    sendbernar::blackbox::Uid uid;
    sendbernar::blackbox::KarmaValue karmaValue;
    sendbernar::blackbox::KarmaStatus karmaStatus;
    std::vector<sendbernar::blackbox::Address> addressList;
    sendbernar::blackbox::DbFields dbfields;
    Aliases aliases;
};
}
}

YREFLECTION_ADAPT_ADT(sendbernar::blackbox::User,
    AS_IS(sendbernar::blackbox::Uid, uid)
    CHANGE_NAME(sendbernar::blackbox::KarmaValue, karma, karmaValue)
    CHANGE_NAME(sendbernar::blackbox::KarmaStatus, karma_status, karmaStatus)
    CHANGE_NAME(std::vector<sendbernar::blackbox::Address>, address-list, addressList)
    AS_IS(sendbernar::blackbox::DbFields, dbfields)
    AS_IS(sendbernar::blackbox::Aliases, aliases)
)

BOOST_FUSION_DEFINE_STRUCT((sendbernar)(blackbox), Response,
    (std::vector<sendbernar::blackbox::User>, users)
)

BOOST_FUSION_DEFINE_STRUCT((sendbernar)(blackbox), Error,
    (std::string, error)
)

BOOST_FUSION_DEFINE_STRUCT((sendbernar)(blackbox), NonexistingUser,
    (std::string, id)
)

BOOST_FUSION_DEFINE_STRUCT((sendbernar)(blackbox), NonexistingAccount,
    (std::vector<sendbernar::blackbox::NonexistingUser>, users)
)

#undef CHANGE_NAME
#undef AS_IS

namespace sendbernar {

const std::string MALISH = "12";

std::variant<std::string, Account> BlackBox::parse(const std::string& json) {
    Account account;
    sendbernar::blackbox::Response response;

    try {
        response = yamail::data::deserialization::fromJson<sendbernar::blackbox::Response>(json);
    } catch (const std::exception&) {
        try {
            auto err = yamail::data::deserialization::fromJson<sendbernar::blackbox::Error>(json);
            return err.error;
        } catch (const std::exception&) { }

        try {
            yamail::data::deserialization::fromJson<sendbernar::blackbox::NonexistingAccount>(json);
            return "No such account for uid";
        } catch (const std::exception&) { }

        return "Unknown error";
    }

    if (response.users.size() != 1) {
        return std::string("There are too many accounts in the response: ") + std::to_string(response.users.size());
    }

    blackbox::User& u = response.users[0];

    if (u.dbfields.suid.empty()) {
        return "No such mail account";
    }

    account.uid = u.uid.value;
    account.language = u.dbfields.lang;
    account.country = u.dbfields.country;
    account.karmaStatus = u.karmaStatus.value;
    account.karmaValue = u.karmaValue.value;
    account.timezone = u.dbfields.tz;

    for (const sendbernar::blackbox::Address& address : u.addressList) {
        if (address.default_) {
            account.defaultAddress = idna::decode_addrspec(address.address);
        }

        if (address.native) {
            account.addresses.emplace_back(idna::decode_addrspec(address.address));
        } else if (address.validated) {
            account.validated.emplace_back(idna::decode_addrspec(address.address));
        }
    }

    const bool malish = u.aliases.count(MALISH) && !u.aliases.at(MALISH).empty();
    split_domain(idna::decode_addrspec(malish ? account.defaultAddress : u.dbfields.login),
                 account.login,
                 account.domain);

    account.fromName = u.dbfields.firstname.empty() || u.dbfields.lastname.empty()
            ? u.dbfields.firstname + u.dbfields.lastname
            : u.dbfields.firstname + " " + u.dbfields.lastname;    

    return account;
}

Account BlackBox::getAccount(const params::CommonParams& params) const {
    const std::vector<std::string> dbFields = {
        "account_info.country.uid",
        "account_info.tz.uid",
        "subscription.login.2",
        "subscription.suid.2",
        "userinfo.lang.uid",
        "userinfo.firstname.uid",
        "userinfo.lastname.uid"
    };

    HttpArguments args;
    args.add("uid", params.uid);
    args.add("userip", params.realIp);
    args.add("host", authDomain);
    args.add("dbfields", boost::algorithm::join(dbFields, ","));
    args.add("aliases", MALISH);
    args.add("method", "userinfo");
    args.add("emails", "getall");
    args.add("format", "json");

    std::variant<std::string, Account> response("cannot get account");

    http.req(http.toGET(bb).getArgs("args"_arg=args))->call(EndpointType::blackbox, withDefaultHttpWrap(
        [&](yhttp::response resp) {
        response = BlackBox::parse(resp.body);
    }));

    if (const std::string* err = std::get_if<std::string>(&response)) {
        throw BlackBoxException(*err);
    } else if (const Account* acc = std::get_if<sendbernar::Account>(&response)) {
        return *acc;
    } else {
        throw BlackBoxException("cannot get error or account");
    }
}

}
