#include <mail/sendbernar/services/include/delayed.h>
#include <mail/sendbernar/core/include/account.h>
#include <mail/sendbernar/services/include/reminders.h>
#include <butil/datetime/date_utils.h>
#include <yamail/data/deserialization/json_reader.h>
#include <mail/sendbernar/client/include/category.h>


BOOST_FUSION_DEFINE_STRUCT((sendbernar), Callback400Response,
    (std::string, status)
)

namespace sendbernar {

mail_errors::error_code createDelayedReminder(const params::CommonParams& common, const params::UserJournalParams& uj,
                                              const params::SendDelayed& delayed, params::ContinueSendingMessage params,
                                              const SourceMid& sourceMid, const http_getter::TypedClient& http, SendConfigurationPtr ptr,
                                              ContextLogger logger) {

    const std::time_t sendDate = delayedMessageSendDate(delayed, ptr->reminders().sendUndoAdditionalTime);
    const std::string reminderDate = callmeback::Client::formatDate(sendDate);

    request::Request req = request::callbackDelayedOrUndoMessage(common, uj, params, delayed.relative);

    RemindersType type = delayed.relative
            ? RemindersType::undo
            : RemindersType::delayed;

    LOGDOG_(logger, notice, log::message="reminder_params",
            log::mid=params.mid, log::reminder_type=type, log::reminder_date=reminderDate);

    const bool update = params.mid == sourceMid.value() && sourceMid.isDelayedOrUndo();

    bool result = false;
    callmeback::Client(ptr->reminders().clientConfiguration, ptr->wmiYplatformServer(), http)
            .set(common.uid, params.mid, type, reminderDate, req, update)
            ->call(EndpointType::setting_delayed_reminder, [&] (yhttp::response resp) {
        unsigned status = resp.status;
        if (http_getter::helpers::successCode(status)) {
            result = true;
            return http_getter::Result::success;
        } else if (status == 400) {
            const auto r = yamail::data::deserialization::fromJson<Callback400Response>(resp.body);

            if (r.status == "already_exists") {
                result = true;
                return http_getter::Result::success;
            } else {
                return http_getter::Result::retry;
            }
        } else {
            return http_getter::helpers::retriableCode(status) ? http_getter::Result::retry : http_getter::Result::fail;
        }
    });

    if (!result) {
        return make_error(ErrorResult::reminderError, "cannot create delayed reminder");
    }

    return mail_errors::error_code();
}

void deleteDelayedReminder(const std::string& uid, const SourceMid& sourceMid, const http_getter::TypedClient& http,
                           SendConfigurationPtr configuration) {
    if (sourceMid.isDelayedOrUndo()) {
        RemindersType type = sourceMid.isUndo()
                ? RemindersType::undo
                : RemindersType::delayed;

        callmeback::Client(configuration->reminders().clientConfiguration, configuration->wmiYplatformServer(), http)
                .remove(uid, type, sourceMid.value())
                ->backgroundCall(EndpointType::remove_delayed_reminder,
                            withDefaultHttpWrap([](yhttp::response) {}));
    }
}

mail_errors::error_code deleteDelayedReminderSync(const std::string& uid, const SourceMid& sourceMid, const http_getter::TypedClient& http,
                                                  SendConfigurationPtr configuration) {
    if (sourceMid.isDelayedOrUndo()) {
        ErrorResult errorCode = ErrorResult::reminderError;

        RemindersType type = sourceMid.isUndo()
                ? RemindersType::undo
                : RemindersType::delayed;

        callmeback::Client(configuration->reminders().clientConfiguration, configuration->wmiYplatformServer(), http)
                .remove(uid, type, sourceMid.value())
                ->call(EndpointType::remove_delayed_reminder, withDefaultHttpWrap([&](yhttp::response) {
            errorCode = ErrorResult::ok;
        }));

        if (errorCode != ErrorResult::ok) {
            return mail_errors::error_code(errorCode, "cannot remove delayed reminder");
        }
    }

    return mail_errors::error_code();
}

}
