#pragma once

#include <mail/sendbernar/core/include/configuration.h>
#include <mail/sendbernar/core/include/recipients_repository.h>
#include <mail/sendbernar/core/include/composed_message.h>
#include <mail/http_getter/client/include/client.h>
#include <mail/sendbernar/core/include/metadata.h>
#include <mail/sendbernar/core/include/account.h>

namespace sendbernar {

template<class EmailRepo>
bool checkMentionsInRecipients(const boost::optional<params::Mentions>& mentions, const EmailRepo& repo) {
    if (!mentions) {
        return true;
    }

    EmailVec vec = repo.toVector();
    std::set<std::string> recipients;

    boost::copy(vec
                | boost::adaptors::transformed([] (const Email& email) { return email.addressString(); })
                , std::inserter(recipients, recipients.begin()));

    for (const params::Mention& m : *mentions) {
        if (recipients.find(m) == recipients.end()) {
            return false;
        }
    }

    return true;
}

template<class Params>
std::vector<macs::Lid> filteredLids(const Params& params, MetadataPtr meta) {
    std::vector<macs::Lid> ret;

    if (params.lids) {
        const macs::Label delayed = meta->labelBySymbol(macs::Label::Symbol::delayedMessage_label);
        const macs::Label undo = meta->labelBySymbol(macs::Label::Symbol::undoMessage_label);
        const macs::Label draft = meta->labelBySymbol(macs::Label::Symbol::draft_label);

        boost::copy(*params.lids
                    | boost::adaptors::filtered([&](const std::string& lid) {
                        return lid != draft.lid() && lid != undo.lid() && lid != delayed.lid();
                    })
                    , std::back_inserter(ret)
        );
    }

    return ret;
}

using SaveResult = std::pair<DeliveryResult, macs::Mid>;
using PersonalLabels = std::map<sendbernar::params::Mention, std::vector<std::string>>;

std::pair<http_getter::Result, SendResult> makeSendStatus(yhttp::response resp);
std::pair<http_getter::Result, SaveResult> makeSaveStatus(yhttp::response resp);

struct NwSmtp {
private:
    const http_getter::TypedClient& http;
    RequestMertics metrics;
    const SendConfiguration::NwSmtp& cfg;
    params::CommonParams common;

    template<class Message>
    SendResult sendImpl(const http_getter::TypedEndpoint& endpoint, bool checkSpam, bool notifyOnSend, const macs::Lids& lids,
                            const boost::optional<macs::Mid>& sourceMid, const boost::optional<params::Mentions>& mentions,
                            Message message) const;

    SaveResult saveImpl(const std::vector<macs::Label::Symbol>& symbols,
                        const macs::Lids& lids,
                        const macs::Fid& fid,
                        const boost::optional<macs::Mid>& sourceMid,
                        std::string realMailbox, boost::shared_ptr<std::string> text) const;

public:
    NwSmtp(const params::CommonParams& common, RequestMertics m, const http_getter::TypedClient& http, const SendConfiguration::NwSmtp& cfg)
            : http(http)
            , metrics(m)
            , cfg(cfg)
            , common(common)
    { }

    SendResult sendSystemMessage(const std::vector<std::string>& labels, const std::vector<macs::Label::Symbol>& symbols,
                                     const macs::Lids& lids, const RecipientsRepository& recipients,
                                     compose::SystemMessage message) const;

    SendResult send(bool notifyOnSend, const macs::Lids& lids,
                        const boost::optional<macs::Mid>& sourceMid, const boost::optional<params::Mentions>& mentions,
                        compose::SendMessage message) const;

    SendResult sendShare(const macs::Lids& lids, compose::SendShare message) const;

    SendResult sendDelayed(bool notifyOnSend, const macs::Lids& lids,
                               const boost::optional<macs::Mid>& sourceMid, const boost::optional<params::Mentions>& mentions,
                               compose::SendMessage message) const;

    template<class ComposeStruct>
    SaveResult save(const std::vector<macs::Label::Symbol>& symbols,
                    const macs::Lids& lids,
                    const macs::Fid& fid,
                    const boost::optional<macs::Mid>& sourceMid,
                    ComposeStruct message) const {
        return saveImpl(symbols, lids, fid, sourceMid, std::move(message.realMailbox), message.text);
    }

    SaveResult saveDelayed(const std::vector<macs::Label::Symbol>& symbols,
                           const macs::Lids& lids,
                           const macs::Fid& fid,
                           const boost::optional<macs::Mid>& sourceMid,
                           compose::SendDelayed message) const;
};

}
