#include <mail/sendbernar/services/include/postprocess.h>
#include <user_journal/parameters/mailbox.h>
#include <mail/sendbernar/services/include/reminders.h>
#include <butil/datetime/date_utils.h>
#include <butil/StrUtils/utfutils.h>
#include <yplatform/find.h>


namespace sendbernar {

void PostProcessMessage::setNoAnswerReminder() {
    if (!postprocess.lids) {
        return;
    }

    const std::string remindMeLid = metadata->labelBySymbol(macs::Label::Symbol::remindNoAnswer_label).lid();

    if (std::find(postprocess.lids->begin(), postprocess.lids->end(), remindMeLid) != postprocess.lids->end()) {
        try {
            std::time_t remindPeriod = config->send->reminders().noAnswerRemindPeriod;
            if (postprocess.noanswer_remind_period) {
                remindPeriod = *postprocess.noanswer_remind_period;
            }
            const time_t date = postprocess.message_date + remindPeriod;
            const std::string reminderDate = callmeback::Client::formatDate(date);
            const std::string originalDate = DateUtils::tztime2string(postprocess.message_date,
                                                                      account.timezone, "%d.%m.%Y");

            params::NoAnswerRemindCallback p;
            p.date = originalDate;
            p.lang = account.language;
            p.message_id = postprocess.message_id;

            callmeback::Client(config->send->reminders().clientConfiguration, config->send->wmiYplatformServer(), *http)
                    .set(account.uid, p.message_id, RemindersType::noAnswer,
                         reminderDate, request::callbackNoAnswer(common, uj, p), false)
                    ->backgroundCall(EndpointType::setting_noanswer_reminder,
                                withDefaultHttpWrap([] (yhttp::response) {}));
        } catch (const std::exception& e) {
            LOGDOG_(logger, notice, log::message="cannot set no answer reminder", log::exception=e);
        }
    }
}

void PostProcessMessage::logMessageSent() {
    try {
        using namespace user_journal::parameters;

        auto journal = config->maildb->journal(
            serviceParams(common),
            journalParams(uj)
        );

        std::ostringstream state;
        utf::cut(postprocess.nonempty_subject, 25);
        state << "message-id=" << postprocess.message_id << std::endl
              << "subject="    << postprocess.nonempty_subject << std::endl
              << "source_mid=" << postprocess.source_mid.get_value_or("");
        if (postprocess.nonempty_recipients.to) {
            state << std::endl << "to=" << *postprocess.nonempty_recipients.to;
        }
        if (postprocess.nonempty_recipients.cc) {
            state << std::endl << "cc=" << *postprocess.nonempty_recipients.cc;
        }
        if (postprocess.nonempty_recipients.bcc) {
            state << std::endl << "bcc=" << *postprocess.nonempty_recipients.bcc;
        }

        journal.write<SendMessage>(
            id::state(state.str()),
            id::mid(postprocess.source_mid.get_value_or("")),
            id::affected(1ul),
            id::mdb("pg"),
            id::msgId(postprocess.message_id),
            id::emailTo(postprocess.nonempty_recipients.to.get_value_or("")),
            id::emailCc(postprocess.nonempty_recipients.cc.get_value_or("")),
            id::emailBcc(postprocess.nonempty_recipients.bcc.get_value_or(""))
        );
    } catch (const std::exception& e) {
        LOGDOG_(logger, notice, log::message="cannot write to journal", log::exception=e);
    }
}

void PostProcessMessage::markSpecialMids() {
    if (postprocess.inreplyto) {
        const params::InReplyTo& irt = *postprocess.inreplyto;
        try {
            static const macs::Folder::Symbol excludedSymbols[] = { macs::Folder::Symbol::drafts, macs::Folder::Symbol::template_,
                                       macs::Folder::Symbol::outbox, macs::Folder::Symbol::trash,
                                       macs::Folder::Symbol::hidden_trash };
            macs::FidVec fids;
            fids.reserve(std::size(excludedSymbols));
            for (const auto& symbol: excludedSymbols) {
                macs::Fid fid = metadata->fidBySymbol(symbol);
                if (!fid.empty()) {
                    fids.emplace_back(std::move(fid));
                }
            }

            const macs::MidVec mids = metadata->getByMessageId(irt.inreplyto, fids);
            if (mids.size() > 0 && mids.size() <= 2) {
                macs::Label::Symbol symbol = irt.mark_as == params::MarkAs::replied ? macs::Label::Symbol::answered_label
                                                                                    : macs::Label::Symbol::forwarded_label;

                macs::Label label = metadata->labelBySymbol(symbol);

                metadata->markEnvelopes(mids, {label});
            } else {
                LOGDOG_(logger, warning, log::message="strange number of messages by message_id: " + std::to_string(mids.size()));
            }
        } catch (const std::exception& e) {
            LOGDOG_(logger, error, log::message="cannot mark as answered or replied", log::exception=e);
        }
    }

    if (postprocess.forward_mids && !postprocess.forward_mids->empty()) {
        try {
            macs::Label label = metadata->labelBySymbol(macs::Label::Symbol::forwarded_label);

            metadata->markEnvelopes(*postprocess.forward_mids, {label});
        } catch (const std::exception& e) {
            LOGDOG_(logger, error, log::message="cannot mark as forwarded", log::exception=e);
        }
    }
}

void PostProcessMessage::deleteDraft() {
    if (postprocess.source_mid) {
        try {
            metadata->deleteDraft(*postprocess.source_mid);
        } catch (const std::exception& e) {
            LOGDOG_(logger, notice, log::message="Can't delete draft",
                    log::mid=*postprocess.source_mid, log::exception=e);
        } catch (...) {
            LOGDOG_(logger, notice, log::message="Can't delete draft: unknown error",
                    log::mid=*postprocess.source_mid);
        }
    }
}

void PostProcessMessage::make() {
    logMessageSent();
    setNoAnswerReminder();
    markSpecialMids();
    deleteDraft();
}

void PostProcessMessage::make(boost::asio::yield_context yield) {
    metadata = metadata->clone(yield);
    make();
}

template<class Message>
params::PostProcessMessage makePostprocessParamsImpl(const Message& msg, const RecipientsRepository& recipients,
                                                     const macs::Lids& filteredLids, const boost::optional<macs::Mid>& sourceMid,
                                                     const params::SendMessage& params) {
    params::PostProcessMessage postprocess;
    postprocess.nonempty_subject = msg.nonEmptySubject;
    postprocess.message_id = msg.messageId;
    postprocess.message_date = msg.date;
    postprocess.forward_mids = boost::make_optional(!msg.forwardedMids.empty(), msg.forwardedMids);
    postprocess.nonempty_recipients.to = boost::make_optional(recipients.hasToRecipients(), recipients.genToString());
    postprocess.nonempty_recipients.cc = boost::make_optional(recipients.hasCcRecipients(), recipients.genCcString());
    postprocess.nonempty_recipients.bcc = boost::make_optional(recipients.hasBccRecipients(), recipients.genBccString());
    postprocess.lids = boost::make_optional(!filteredLids.empty(), filteredLids);
    postprocess.source_mid = sourceMid;
    postprocess.noanswer_remind_period = params.delivery.noanswer_remind_period;
    postprocess.inreplyto = params.inreplyto;

    return postprocess;
}

params::PostProcessMessage makePostprocessParams(const compose::SendDelayed& msg,
                                                 const macs::Lids& filteredLids, const boost::optional<macs::Mid>& sourceMid,
                                                 const params::SendMessage& params) {
    return makePostprocessParamsImpl(msg, msg.recipients, filteredLids, sourceMid, params);
}

params::PostProcessMessage makePostprocessParams(const compose::SendMessage& msg,
                                                 const macs::Lids& filteredLids, const SourceMid& sourceMid,
                                                 const params::SendMessage& params) {
    return makePostprocessParamsImpl(msg, msg.recipients, filteredLids, sourceMid.optional(), params);
}

}
