#include <mail/sendbernar/services/include/smtp_gate.h>

#include <yamail/data/serialization/json_writer.h>
#include <yamail/data/deserialization/json_reader.h>
#include <ymod_httpclient/errors.h>
#include <butil/datetime/date_utils.h>
#include <butil/datetime/timezone.h>
#include <butil/http/arguments.h>
#include <yplatform/tskv/tskv.h>


BOOST_FUSION_DEFINE_STRUCT((sendbernar), SmtpGateSendResponse,
    (std::string, error)
    (std::optional<std::string>, ban_reason)
)

BOOST_FUSION_DEFINE_STRUCT((sendbernar), SmtpGateSaveResponse,
    (std::string, error)
    (macs::Mid, mid)
)

using PersonalLabels = std::map<sendbernar::params::Mention, std::vector<std::string>>;

BOOST_FUSION_DEFINE_STRUCT((sendbernar), SendMailParams,
    (std::string, ip)
    (std::string, host)
    (std::string, source)

    (std::vector<std::string>, to)
    (std::vector<std::string>, cc)
    (std::vector<std::string>, bcc)
    (std::vector<std::string>, lid)
    (std::vector<std::string>, sender_label)
    (std::vector<std::string>, common_label)

    (boost::optional<std::string>, notify)
    (boost::optional<std::string>, save_to_sent)

    (PersonalLabels, personal_labels)
)

namespace sendbernar {
std::string addSymbolPrefix(const macs::Label::Symbol& symbol) {
    return std::string("symbol:")+symbol.title();
}

std::pair<http_getter::Result, SendResult> makeSendStatus(yhttp::response resp) {
    unsigned httpCode = resp.status;
    switch(httpCode) {
        case 200:
            return std::make_pair(http_getter::Result::success, SendResult(DeliveryResult::ok));
        case 400:
            return std::make_pair(http_getter::Result::fail, SendResult(DeliveryResult::badRequest));
        case 406:
        {
            SendResult ret;
            const SmtpGateSendResponse response = yamail::data::deserialization::fromJson<SmtpGateSendResponse>(resp.body);
            const std::string& reason = response.error;
            if (reason == "Spam") {
                ret.deliveryResult = DeliveryResult::spam;
            } else if (reason == "StrongSpam") {
                ret = SendResult(DeliveryResult::strongSpam, response.ban_reason);
            } else if (reason == "Virus") {
                ret.deliveryResult = DeliveryResult::virus;
            } else if (reason == "BadRecipient") {
                ret.deliveryResult = DeliveryResult::badRecipient;
            } else if (reason == "BadKarmaBanTime") {
                ret.deliveryResult = DeliveryResult::badKarmaBanTime;
            } else if (reason == "SendMessageFailed") {
                ret.deliveryResult = DeliveryResult::sendMessageFailed;
            } else if (reason == "ServiceUnavaliable") {
                ret.deliveryResult = DeliveryResult::serviceUnavaliable;
            } else if (reason == "FailedToAuthSender") {
                ret.deliveryResult = DeliveryResult::failedToAuthSender;
            } else if (reason == "BadSender") {
                ret.deliveryResult = DeliveryResult::badSender;
            } else if (reason == "ToManyRecipients") {
                ret.deliveryResult = DeliveryResult::toManyRecipients;
            } else if (reason == "SizeLimitExceeded") {
                ret.deliveryResult = DeliveryResult::sizeLimitExceeded;
            } else {
                ret.deliveryResult = DeliveryResult::unknownError;
            }
            return std::make_pair(http_getter::Result::fail, ret);
        }; break;

        default:
            return std::make_pair(http_getter::Result::retry, SendResult(DeliveryResult::unknownError));
    }
}

std::pair<http_getter::Result, SaveResult> makeSaveStatus(yhttp::response resp) {
    unsigned httpCode = resp.status;

    switch(httpCode) {
        case 200:
        {
            const std::string mid = yamail::data::deserialization::fromJson<SmtpGateSaveResponse>(resp.body).mid;
            return std::make_pair(http_getter::Result::success, std::make_pair(DeliveryResult::ok, mid == "none" ? "" : mid));
        }; break;
        case 400:
            return std::make_pair(http_getter::Result::fail, std::make_pair(DeliveryResult::badRequest, ""));
        case 406:
        {
            DeliveryResult ret;
            const std::string reason = yamail::data::deserialization::fromJson<SmtpGateSaveResponse>(resp.body).error;
            if (reason == "Spam") {
                ret = DeliveryResult::spam;
            } else if (reason == "StrongSpam") {
                ret = DeliveryResult::strongSpam;
            } else if (reason == "Virus") {
                ret = DeliveryResult::virus;
            } else if (reason == "BadRecipient") {
                ret = DeliveryResult::badRecipient;
            } else {
                ret = DeliveryResult::unknownError;
            }
            return std::make_pair(http_getter::Result::fail, std::make_pair(ret, ""));
        }; break;

        default:
            return std::make_pair(http_getter::Result::retry, std::make_pair(DeliveryResult::unknownError, ""));
    }
}

inline http::headers headers() {
    http::headers h;
    h.add("Content-Type", "text/plain");
    return h;
}

const std::string peopleLabel = "SystMetkaSO:people";

const std::string& extractUid(const params::CommonParams& common, const compose::SendMessage&) {
    return common.uid;
}

const std::string& extractUid(const params::CommonParams&, const compose::SendShare& share) {
    return share.adminUid;
}

const compose::SendMessage& extractMessage(const compose::SendMessage& msg) {
    return msg;
}

const compose::SendMessage& extractMessage(const compose::SendShare& msg) {
    return msg.message;
}

SendResult NwSmtp::sendSystemMessage(const std::vector<std::string>& labels,
                                         const std::vector<macs::Label::Symbol>& symbols,
                                         const macs::Lids& lids,
                                         const RecipientsRepository& recipients,
                                         compose::SystemMessage message) const {
    HttpArguments args;

    args.add("uid", common.uid);
    args.add("from", message.envelopeFrom);
    args.add("request_id", common.requestId);
    args.add("service", "sendbernar");

    boost::range::for_each(recipients.toVector(), [&args](const Email& e) {
        args.add("to", e.addressString());
    });

    boost::range::for_each(labels, [&args] (const std::string& label) {
        args.add("label", label);
    });

    boost::range::for_each(symbols, [&args] (const macs::Label::Symbol& symbol) {
        args.add("label", addSymbolPrefix(symbol));
    });

    boost::range::for_each(lids, [&args](const std::string& lid) {
        args.add("lid", lid);
    });

    SendResult ret(DeliveryResult::unknownError);
    http.req(http.toPOST(cfg.system_send)
                     .getArgs("args"_arg=args)
                     .headers("hdrs"_hdr=headers())
                     .body(message.text))
            ->call(EndpointType::system_send, [&, this](yhttp::response resp) {
                http_getter::Result result;
                std::tie(result, ret) = makeSendStatus(std::move(resp));
                this->metrics.smtpGateStatus(ret.deliveryResult);
                return result;
            });
    return ret;
}

template<class Message>
SendResult NwSmtp::sendImpl(const http_getter::TypedEndpoint& endpoint, bool checkSpam, bool notifyOnSend, const macs::Lids& lids,
                                const boost::optional<macs::Mid>& sourceMid, const boost::optional<params::Mentions>& mentions,
                                Message msg) const {

    const compose::SendMessage& message = extractMessage(msg);

    HttpArguments args;

    if (checkSpam) {
        args.add("detect_spam", "1");
        args.add("detect_virus", "1");
    } else {
        args.add("detect_spam", "0");
        args.add("detect_virus", "0");
    }

    args.add("uid", extractUid(common, msg));
    args.add("from_email", msg.envelopeFrom);
    args.add("request_id", common.requestId);
    args.add("service", "sendbernar");
    if (sourceMid) {
        args.add("old_mid", *sourceMid);
    }

    SendMailParams params;

    params.ip = common.realIp;
    params.host = common.originalHost;
    params.lid = lids;
    params.common_label = { peopleLabel };


    if (mentions) {
        const std::vector<std::string> mentionLabels = {
                addSymbolPrefix(macs::Label::Symbol::mention_label),
                addSymbolPrefix(macs::Label::Symbol::mention_unvisited_label),
        };

        boost::copy(*mentions
                    | boost::adaptors::transformed([&](const params::Mention& mention) { return std::make_pair(mention, mentionLabels); })
                , std::inserter(params.personal_labels, params.personal_labels.begin()));
    }

    boost::copy(message.recipients.to()
                | boost::adaptors::transformed([](const Email& e) { return e.addressString(); })
            , std::back_inserter(params.to));

    boost::copy(message.recipients.cc()
                | boost::adaptors::transformed([](const Email& e) { return e.addressString(); })
            , std::back_inserter(params.cc));

    boost::copy(message.recipients.bcc()
                | boost::adaptors::transformed([](const Email& e) { return e.addressString(); })
            , std::back_inserter(params.bcc));

    if (notifyOnSend) {
        params.notify = "1";
    }

    if (message.saveToSent) {
        params.save_to_sent = "1";

        params.sender_label.push_back(addSymbolPrefix(macs::Label::Symbol::seen_label));

        if (notifyOnSend) {
            params.sender_label.push_back("delivery_confirmation");
        }

        if (message.hasAttachments) {
            params.sender_label.push_back(addSymbolPrefix(macs::Label::Symbol::attached_label));
        }
    }

    const std::string boundary = "send_message_boundary";
    const std::string boundaryStart = "--" + boundary;
    const std::string boundaryEnd = "--" + boundary + "--";
    const std::string crlf = "\r\n";
    std::ostringstream multipart;
    multipart << crlf
              << boundaryStart << crlf
              << "Content-Type: application/json" << crlf
              << crlf
              << yamail::data::serialization::JsonWriter<SendMailParams>(params).result() << crlf
              << boundaryStart << crlf
              << "Content-Type: message/rfc822" << crlf
              << crlf
              << *message.text << crlf
              << boundaryEnd << crlf
            ;

    SendResult ret(DeliveryResult::unknownError);
    http.req(http.toPOST(endpoint)
                     .getArgs("args"_arg=args)
                     .headers("Content-Type"_hdr="multipart/mixed; boundary=" + boundary)
                     .body(multipart.str()))
            ->call(EndpointType::send, [&, this](yhttp::response resp) {
                http_getter::Result result;
                std::tie(result, ret) = makeSendStatus(std::move(resp));
                this->metrics.smtpGateStatus(ret.deliveryResult);
                return result;
            });
    return ret;
}

SendResult NwSmtp::sendDelayed(bool notifyOnSend, const macs::Lids& lids,
                                   const boost::optional<macs::Mid>& sourceMid, const boost::optional<params::Mentions>& mentions,
                                   compose::SendMessage message) const {
    return sendImpl(cfg.delayed_callback_send, false, notifyOnSend, lids, sourceMid, mentions, std::move(message));
}

SendResult NwSmtp::send(bool notifyOnSend, const macs::Lids& lids,
                            const boost::optional<macs::Mid>& sourceMid, const boost::optional<params::Mentions>& mentions,
                            compose::SendMessage message) const {
    return sendImpl(cfg.send, true, notifyOnSend, lids, sourceMid, mentions, std::move(message));
}

SendResult NwSmtp::sendShare(const macs::Lids& lids, compose::SendShare message) const {
    return sendImpl(cfg.send, true, false, lids, boost::none, boost::none, std::move(message));
}

SaveResult NwSmtp::saveImpl(const std::vector<macs::Label::Symbol>& symbols,
                                    const macs::Lids& lids,
                                    const macs::Fid& fid,
                                    const boost::optional<macs::Mid>& sourceMid,
                                    std::string realMailbox,
                                    boost::shared_ptr<std::string> text) const {
    HttpArguments args;

    args.add("uid", common.uid);
    args.add("email", realMailbox);
    args.add("fid", fid);
    args.add("request_id", common.requestId);
    args.add("service", "sendbernar");

    if (sourceMid) {
        args.add("old_mid", *sourceMid);
    }

    boost::range::for_each(symbols, [&args] (const macs::Label::Symbol& symbol) {
        args.add("symbol", addSymbolPrefix(symbol));
    });

    args.add("system", peopleLabel);

    boost::range::for_each(lids, [&args](const std::string& lid) {
        args.add("lid", lid);
    });

    SaveResult ret(DeliveryResult::unknownError, "");
    http.req(http.toPOST(cfg.save)
                     .getArgs("args"_arg=args)
                     .headers("hdrs"_hdr=headers())
                     .body(text))
            ->call(EndpointType::save, [&, this](yhttp::response resp) {
                http_getter::Result result;
                std::tie(result, ret) = makeSaveStatus(std::move(resp));
                this->metrics.smtpGateStatus(ret.first);
                return result;
            });
    return ret;
}

SaveResult NwSmtp::saveDelayed(const std::vector<macs::Label::Symbol>& symbols,
                                       const macs::Lids& lids,
                                       const macs::Fid& fid,
                                       const boost::optional<macs::Mid>& sourceMid,
                                       compose::SendDelayed message) const {
    HttpArguments args;

    args.add("uid", common.uid);
    args.add("email", message.realMailbox);
    args.add("fid", fid);
    args.add("request_id", common.requestId);
    args.add("received_date", std::to_string(message.sendDate));
    args.add("service", "sendbernar");

    if (sourceMid) {
        args.add("old_mid", *sourceMid);
    }

    boost::range::for_each(symbols, [&args] (const macs::Label::Symbol& symbol) {
        args.add("symbol", addSymbolPrefix(symbol));
    });

    args.add("system", peopleLabel);

    boost::range::for_each(lids, [&args](const std::string& lid) {
        args.add("lid", lid);
    });

    SaveResult ret(DeliveryResult::unknownError, "");
    http.req(http.toPOST(cfg.save_delayed)
                     .getArgs("args"_arg=args)
                     .headers("hdrs"_hdr=headers())
                     .body(message.text))
            ->call(EndpointType::save_delayed, [&, this](yhttp::response resp) {
                http_getter::Result result;
                std::tie(result, ret) = makeSaveStatus(std::move(resp));
                this->metrics.smtpGateStatus(ret.first);
                return result;
            });
    return ret;
}

}

