#pragma once

#include <variant>

#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>

#include <mail/unistat/cpp/include/run.h>
#include <mail/unistat/cpp/include/common_logs.h>

#include <mail/sendbernar/metrics/metrics.h>

#include <util/generic/typetraits.h>


namespace sendbernar::metrics {

struct Metric {
    std::map<std::string, std::size_t> accum;

    Metric() = default;

    static std::string fullName(const std::string& version, const std::string& name) {
        return "ctype="+unistat::normalizeName(version)+";"+name;
    }

    void inc(const std::string& version, const std::string& name) {
        const std::string full = fullName(version, name);
        if (accum.find(full) == accum.end()) {
            accum[full] = 1;
        } else {
            accum[full]++;
        }
    }

    void update(const std::map<std::string, std::string>& record) {
        using namespace std::string_literals;
        namespace yr = yamail::data::reflection;

        const auto it = record.find(logdog::name(log::version));
        if (it == record.end()) {
            throw std::runtime_error("missing version argument");
        }

        const std::string version = it->second;
        std::visit([&, this] (const auto& m) {
            using T = std::decay_t<decltype(m)>;

            if constexpr (std::is_same_v<T, ServiceRequest>) {
                const std::string status(to_string(m.status));

                inc(version, m.operation + "_"s + status);

                std::size_t i = 0;
                for (; i < m.calls.size()-1; i++) {
                    inc(version, m.operation + "_try_"s + std::to_string(i) + "_fail"s);
                }

                inc(version, m.operation + "_try_"s + std::to_string(i) + "_success"s);
            } else if constexpr (std::is_same_v<T, ErrorCode>) {
                inc(version, m.category + "_"s + m.message + "_"s + std::to_string(m.value));
            } else if constexpr (std::is_same_v<T, SmtpGate>) {
                inc(version, std::string(yr::to_string(m.result)));
            } else if constexpr (std::is_same_v<T, std::monostate>) {
                throw std::runtime_error("cannot deserialize");
            } else {
                static_assert(TDependentFalse<T>, "non-exhaustive visitor");
            }
        }, deserialize(record));
    }

    std::vector<unistat::NamedValue<std::size_t>> get() const {
        std::vector<unistat::NamedValue<std::size_t>> ret;
        ret.reserve(accum.size());

        boost::copy(accum
                    | boost::adaptors::transformed([](const auto& p) {
                        return unistat::NamedValue<std::size_t>{unistat::withSigoptSuffix(p.first, "summ"), p.second};
                    })
                    , std::back_inserter(ret));

        return ret;
    }
};

}

namespace unistat {

using Metric = sendbernar::metrics::Metric;
using LogMeters = std::variant<std::shared_ptr<sendbernar::metrics::Metric>>;
using MetricsLog = Log<TextFileReader, File, TskvParser, LogMeters>;
using MetricsLogPtr = std::shared_ptr<MetricsLog>;
using Logs = std::variant<MetricsLogPtr, HttpClientLogPtr>;

}
