from argparse import ArgumentParser
from mail.unistat.cpp.cython.meters import (
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
)
from collections import namedtuple
import os
import yaml
import logging
import mail.unistat.cpp.cython.logs as logs
import mail.sendbernar.unistat.cpp.run as unistat


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/var/log/sendbernar/unistat.log', help='path for yplatform.log')
    parser.add_argument('sendbernar_config')
    return parser.parse_args()


def remove_scheme(s):
    return s.replace('https://', '').replace('http://', '')


def remove_scheme_and_port(s):
    s = remove_scheme(s)
    i = s.rfind(':')
    if i != -1:
        return s[:i]
    return s


def endpoint_hosts(ep):
    return ep['client']['nodes']


def endpoint_url(ep):
    return endpoint_hosts(ep) + ep['method']


def make_sendbernar_config(data):
    sbr = list(filter(lambda m: m['_name'] == 'sendbernar', data['config']['modules']['module']))[0]
    send_cfg = sbr['configuration']['send']
    nwsmtp = send_cfg['nwsmtp']
    storage_url = remove_scheme_and_port(endpoint_hosts(send_cfg['mulcagate']['endpoint']))
    sanitizer_url = remove_scheme_and_port(endpoint_hosts(send_cfg['sanitizer']['endpoint']))

    return SendbernarConfig(
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
        metrics_log=os.path.join(os.curdir, data['config']['log']['metrics']['sinks'][0]['path']),
        sanitizer_url=sanitizer_url,
        storage_url=storage_url,
        nwsmtp_save_url=remove_scheme(endpoint_url(nwsmtp['save'])),
        nwsmtp_send_url=remove_scheme(endpoint_url(nwsmtp['send'])),
        nwsmtp_system_send_url=remove_scheme(endpoint_url(nwsmtp['system_send'])),
    )


def make_sendbernar_http_client_log_meters(cfg):
    return [
        HttpClientHttpRequestCountByStatus(cfg.sanitizer_url, "count_by_status_sanitizer"),
        HttpClientHttpRequestCountByStatus(cfg.storage_url, "count_by_status_storage"),
        HttpClientHttpRequestCountByStatus(cfg.nwsmtp_save_url, "count_by_status_nwsmtp_save"),
        HttpClientHttpRequestCountByStatus(cfg.nwsmtp_send_url, "count_by_status_nwsmtp_send"),
        HttpClientHttpRequestCountByStatus(cfg.nwsmtp_system_send_url, "count_by_status_nwsmtp_system_send"),

        HttpClientHttpRequestTotalTimeHist((0, 20, 100, 300, 1000), cfg.sanitizer_url, "sanitizer_time_hist"),
        HttpClientHttpRequestTotalTimeHist((0, 100, 500, 1000, 5000), cfg.storage_url, "storage_time_hist"),
        HttpClientHttpRequestTotalTimeHist((0, 100, 500, 1000, 5000, 10000), cfg.nwsmtp_save_url, "nwsmtp_save_time_hist"),
        HttpClientHttpRequestTotalTimeHist((0, 100, 500, 1000, 5000, 10000), cfg.nwsmtp_send_url, "nwsmtp_send_time_hist"),
        HttpClientHttpRequestTotalTimeHist((0, 100, 500, 1000, 5000, 10000), cfg.nwsmtp_system_send_url, "nwsmtp_system_send_time_hist"),
    ]


def make_metrics_log_meters():
    return [
        unistat.SendbernarMetric()
    ]


SendbernarConfig = namedtuple('SendbernarConfig', (
    'httpclient_log',
    'metrics_log',
    'sanitizer_url',
    'storage_url',
    'nwsmtp_save_url',
    'nwsmtp_send_url',
    'nwsmtp_system_send_url',
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.sendbernar_config) as f:
        sendbernar_config = make_sendbernar_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    metrics_log = unistat.SendbernarMetricsLog(
        [],
        make_metrics_log_meters(),
        fast_forward,
        sendbernar_config.metrics_log
    )
    http_client_log = logs.HttpClientLog(
        [],
        make_sendbernar_http_client_log_meters(sendbernar_config),
        fast_forward,
        sendbernar_config.httpclient_log
    )

    unistat.run(args.host, args.port, [metrics_log, http_client_log], args.log, logLevel='info')

if __name__ == '__main__':
    main()
