#!/usr/bin/env python3

from ci._types import *
from lib.aux import *
from lib.retry import *
from lib.docker_registry import *

import yaml
import requests

import json
import os.path
import sys
import subprocess
import datetime
import argparse
import time
from typing import Dict
from types import SimpleNamespace


class DeployInfo(SimpleNamespace):
    finished: bool = False
    version: str = ""
    success: bool = False
    need_retry: bool = False
    debug: str = ""
    extra: str = ""


class Qloud:
    def __init__(self, config: dict, apps: dict, registry: DockerRegistry):
        self.config = config
        self.apps = apps
        self.registry = registry
        self.config["urls"]["upload"] = self.config["urls"]["upload"].format(
            self.config["target_state"]
        )

    def _check_args(self, app: str, env: str = None):
        res = app in self.apps
        if not res:
            raise Exception("no config for app {0}".format(app))
        if res and (env is not None):
            res = res and (env in self.apps[app]["envs"])
            if not res:
                raise Exception("no environment {1} for app {0}".format(app, env))
        return res

    def apply(self, qloud_conf: dict, task: Task, step: DeployStep):
        self._check_args(step.app, step.env)

        comment = task.comment + "\n\n"
        for raw_component in step.components:
            component = ComponentDiff(*raw_component)
            for item in qloud_conf["components"]:
                if item["componentName"] != component.name:
                    continue
                if self.apps[step.app]["registry_path"] in item["properties"]["repository"]:
                    # prefix = item['properties']['repository'].split(':')[0]
                    item["properties"]["repository"] = task.new_docker_image
                    item["properties"]["hash"] = task.new_docker_image_sha256
                    comment += component.name + "\n\n"
                    comment += component.changelog
                    comment += "\n"
        qloud_conf["comment"] = comment
        return qloud_conf

    def get_dump(self, app: str, env: str, version: str = ""):
        self._check_args(app, env)

        url = (
            self.config["urls"]["dump"]
            + "."
            + self.apps[app]["project"]
            + "."
            + env
            + "/"
            + version
        )
        data = http_get(url, self.config["headers"]).json()
        return data

    def get_status(self, app: str, env: str, version: str = ""):
        self._check_args(app, env)

        suffix = "/" + version if version != "" else ""
        url = self.config["urls"]["status"] + "." + self.apps[app]["project"] + "." + env + suffix
        data = http_get(url, self.config["headers"]).json()
        return data

    def get_tasks(self, app: str, env: str, version: str = ""):
        self._check_args(app, env)

        suffix = "/" + version if version != "" else ""
        url = self.config["urls"]["tasks"] + "." + self.apps[app]["project"] + "." + env + suffix
        data = http_get(url, self.config["headers"]).json()
        return data

    def get_instances_info(self, app: str, env: str, version: str = ""):
        self._check_args(app, env)

        suffix = "/" + version if version != "" else ""
        url = self.config["urls"]["status"] + "." + self.apps[app]["project"] + "." + env + suffix
        data = http_get(url, self.config["headers"]).json()

        summary: Dict[str, int] = {}
        for component in data["components"]:
            instances = data["components"][component]["runningInstances"]
            for instance in instances:
                key = instance["status"]
                if not key in summary:
                    summary[key] = 0
                summary[key] += 1
        return summary

    def get_last_version(self, app: str, env: str):
        self._check_args(app, env)

        url = self.config["urls"]["status"] + "." + self.apps[app]["project"] + "." + env
        data = http_get(url, self.config["headers"]).json()
        versions = data["versions"]
        return versions[0] if len(versions) else None

    def find_versions(self, dump: dict, task: Task, step: DeployStep) -> bool:
        for dump_component in dump["components"]:
            for raw_component in step.components:
                component = ComponentDiff(*raw_component)
                if dump_component["componentName"] != component.name:
                    continue
                if dump_component["properties"]["repository"] != task.new_docker_image:
                    return False

        return True

    def create_cert(self, secret_name, cert):
        url = self.config["urls"]["secret"]
        req = {
            "name": secret_name,
            "description": "",
            "type": "server_certificate",
            "content": cert,
        }
        res = http_put_json(url, self.config["headers"], req).json()
        return res

    def get_secrets(self):
        url = self.config["urls"]["secret"]
        res = http_get(url, self.config["headers"]).json()
        return res

    def secret_exists(self, secret_name):
        secrets = self.get_secrets()
        for secret in secrets:
            if secret["name"] == secret_name:
                return True
        return False

    @retry(tries=60, delay=10)
    def get_secret_grants(self, secret_name):
        url = "{}/secret.{}".format(self.config["urls"]["access"], secret_name)
        res = http_get(url, self.config["headers"]).json()
        return res

    @retry(tries=60, delay=10)
    def get_secret_clients(self, secret_name):
        url = "{}/secret.{}/clients".format(self.config["urls"]["secret"], secret_name)
        res = http_get(url, self.config["headers"]).json()
        return res

    @retry(tries=60, delay=10)
    def add_grant_to_secret(self, grant):
        url = "{}".format(self.config["urls"]["grants"])
        http_post_json(url, self.config["headers"], grant)

    @retry(tries=60, delay=10)
    def add_client_to_secret(self, secret_name, client):
        url = "{}/secret.{}/allow".format(self.config["urls"]["secret"], secret_name)
        req = {"objectId": client}
        http_post_urlencoded(url, self.config["headers"], req)

    def get_deploy_status(self, task: Task, step: DeployStep):
        last_known = self.get_last_version(step.app, step.env)
        dump = self.get_dump(step.app, step.env, str(last_known["version"]))

        current = last_known["status"]
        target = last_known["targetState"]

        debug = json.dumps((current, target, last_known))

        extra = json.dumps(
            (
                current,
                self.get_instances_info(task.app, step.env, str(last_known["version"])),
                target,
            )
        )

        if target == "PENDING_DEACTIVATE":
            return DeployInfo(finished=True, success=False, debug=debug)

        if current == "FAILED":
            return DeployInfo(finished=True, success=False, need_retry=True, debug=debug)

        if (
            current in ("DEPLOYING", "DEPLOYED")
            and target in ("DEPLOYING", "DEPLOYED")
            and self.find_versions(dump, task, step)
        ):
            return DeployInfo(finished=True, success=True)

        if (
            current in ("DEPLOYING", "DEPLOYED")
            and target in ("DEPLOYING", "DEPLOYED")
            and not self.find_versions(dump, task, step)
        ):
            return None

        if current in ("NEW", "CONFIGURING") and not self.find_versions(dump, task, step):
            return DeployInfo(
                finished=True, success=False, debug=debug, info="detected another deployment"
            )

        if target in ("DEPLOYING", "DEPLOYED") and self.find_versions(dump, task, step):
            return DeployInfo(finished=False, extra=extra)

        return None

    def run_deploy(self, task: Task, step: DeployStep):
        dump = self.get_dump(step.app, step.env)
        url = self.config["urls"]["upload"]
        res = http_post_json(url, self.config["headers"], self.apply(dump, task, step)).json()
        return res["version"] if res else None

    def run_deploy_from_dump(self, dump):
        url = self.config["urls"]["upload"]
        res = http_post_json(url, self.config["headers"], dump).json()
        return res["version"] if res else None
