#!/usr/bin/env python3

from enum import IntEnum
from typing import Any, NamedTuple, Tuple, List
import json
import time


class TaskStatus(IntEnum):
    nobuild = 1
    active = 2
    finished = 3
    failed = 4
    canceled = 5


class ComponentDiff(NamedTuple):
    name: str
    old_version: str
    changelog: str
    prev_cloud_env_version: str = ""
    changelog_pretty: str = ""


class DeployStep(NamedTuple):
    app: str
    env: str
    components: List[ComponentDiff]
    requires_pause_after: bool = False


class Task(NamedTuple):
    id: str
    app: str
    branch: str
    revision: str
    steps: List[DeployStep]
    next_step: int = 0
    status: TaskStatus = TaskStatus.nobuild

    author: str = ""
    init_ts: int = int(time.time())

    build_number: str = ""
    build_attempt: int = 0
    new_docker_image: str = ""
    new_docker_image_sha256: str = ""
    build_ts: int = int(time.time())

    update_ts: int = int(time.time())

    comment: str = ""

    @staticmethod
    def load(source: str):
        task = Task(*source)._replace(steps=[])
        tmp = Task(*source)
        for s in tmp.steps:
            step = DeployStep(*s)._replace(components=[])
            for c in DeployStep(*s)._asdict()["components"]:
                step.components.append(ComponentDiff(*c))
            task.steps.append(step)
        return task

    def update(self, **kwargs: Any):
        kwargs["update_ts"] = time.time()
        return self._replace(**kwargs)


class QueueItem(NamedTuple):
    plan: Task
    updated_ts: int = int(time.time())
    comment: str = ""
