#!/usr/bin/env python3

import requests
import string
import json
import time
from types import SimpleNamespace


class BuildInfo(SimpleNamespace):
    success: bool = False
    finished: bool = False
    docker_image: str = ""


class Jenkins:
    def __init__(self, config: dict, apps: dict):
        self.config = config
        self.apps = apps

    def get_next_build_num(self, app):
        url = "{jenkins_url}/job/{prj}/api/json?pretty=true".format(
            jenkins_url=self.config["url"], prj=self.apps[app]["jenkins_path"]
        )
        resp = requests.get(
            url,
            auth=requests.auth.HTTPBasicAuth(self.config["user"], self.config["token"]),
            verify=False,
        )
        if resp.status_code != 200:
            return None
        resp_json = json.loads(resp.text)
        return resp_json["nextBuildNumber"]

    def start_build(self, app, branch):
        build_num = self.get_next_build_num(app)
        data = "json=%7B%22parameter%22%3A+%5B%7B%22name%22%3A+%22BRANCH%22%2C+%22value%22%3A+%22{branch}%22%7D%2C+%7B%22name%22%3A+%22CLEAN_WORKING_DIR%22%2C+%22value%22%3A+false%7D%5D%2C+%22Jenkins-Crumb%22%3A+%221e7a29ba77c30716492b2f8ebec85eda%22%7D".format(
            branch=branch
        )
        url = "{jenkins_url}/job/{prj}/build".format(
            jenkins_url=self.config["url"], prj=self.apps[app]["jenkins_path"]
        )
        resp = requests.post(
            url,
            auth=requests.auth.HTTPBasicAuth(self.config["user"], self.config["token"]),
            verify=False,
            data=data,
            headers={"Content-Type": "application/x-www-form-urlencoded"},
        )
        if resp.status_code // 100 != 2:
            raise Exception("Start build error", resp.status_code, json.dumps(resp.text))
        return build_num  # TODO verify build_num

    def get_build_info(self, app, build_num):
        if build_num == "":
            return None
        ret = BuildInfo()
        info = self.get_build_info_raw(app, build_num)
        if info is None:
            ret.finished = False
            return ret
        ret.finished = not info["building"]
        if ret.finished:
            ret.success = info["result"] == "SUCCESS"
            if ret.success:
                ret.docker_image = info["description"].split(" ")[0]
        return ret

    def get_build_info_raw(self, app, build_num):
        url = "{jenkins_url}/job/{prj}/{build_num}/api/json?pretty=true".format(
            jenkins_url=self.config["url"], prj=self.apps[app]["jenkins_path"], build_num=build_num
        )
        resp = requests.get(
            url,
            auth=requests.auth.HTTPBasicAuth(self.config["user"], self.config["token"]),
            verify=False,
        )
        if resp.status_code != 200:
            return None
        resp_json = json.loads(resp.text)
        return resp_json

    def get_done_percent(self, info):
        return min(1.0, (time.time() * 1000.0 - info["timestamp"]) / info["estimatedDuration"])

    def get_log(self, app, build_num):
        url = "{jenkins_url}/job/{prj}/{build_num}/logText/progressiveText?start=0".format(
            jenkins_url=self.config["url"], prj=self.apps[app]["jenkins_path"], build_num=build_num
        )
        resp = requests.get(
            url,
            auth=requests.auth.HTTPBasicAuth(self.config["user"], self.config["token"]),
            verify=False,
        )
        if resp.status_code != 200:
            return None
        log = "".join(filter(lambda c: c in string.printable, resp.text))
        return log.split("\n")

    def _get_crumb(self):
        url = "{jenkins_url}/crumbIssuer/api/json".format(jenkins_url=self.url)
        resp = requests.get(
            url, auth=requests.auth.HTTPBasicAuth(self.user, self.token), verify=False
        )
        if resp.status_code != 200:
            raise Exception("Get crumb error: {} {}".format(resp.status_code, resp.text))
        return json.loads(resp.text)["crumb"]
