#!/usr/bin/env python3

import datetime
import json
import os.path
import time
from filelock import FileLock, Timeout  # type: ignore


class TasksStorage:
    def __init__(self, config):
        self.impl = LocalStorage(config)

    def get_queue(self):
        return self._get_list("tasks.queue")

    def get_canceled(self):
        return self._get_list("tasks.canceled")

    def get_paused(self):
        return self._get_list("tasks.paused")

    def get_finished(self):
        return self._get_list("tasks.finished")

    def set_queue(self, value):
        return self._set_list("tasks.queue", value)

    def set_canceled(self, value):
        return self._set_list("tasks.canceled", value)

    def set_paused(self, value):
        return self._set_list("tasks.paused", value)

    def set_finished(self, value):
        return self._set_list("tasks.finished", value)

    def _get_list(self, name):
        return self.impl.get(name, list())

    def _set_list(self, key, value):
        self.impl.set(key, value)


class LocalStorage:
    def __init__(self, config):
        self.path = os.path.expanduser(config["path"])
        self.lock_file = os.path.expanduser(config["lock_file"])
        if os.path.exists(self.path) and not os.path.isfile(self.path):
            raise Exception("invalid storage path")
        self.timeout = 20

    def get(self, key, defval=None):
        path_suffix = "_" + key
        lock = FileLock(self.lock_file + path_suffix)
        try:
            with lock.acquire(timeout=self.timeout):
                if not os.path.isfile(self.path + path_suffix):
                    return defval
                with open(self.path + path_suffix, "r") as f:
                    data = f.read().strip()
                    return json.loads(data) if data else defval
        except Timeout:
            raise Exception("storage is busy")

    def set(self, key, val):
        path_suffix = "_" + key
        lock = FileLock(self.lock_file + path_suffix)
        try:
            with lock.acquire(timeout=self.timeout):
                with open(self.path + path_suffix, "w") as f:
                    f.write(json.dumps(val))
        except Timeout:
            raise Exception("storage is busy")

    def get_datetime(self, key, defval=None):
        val = self.get(key)
        if val is None:
            return defval
        return defval if val is None else datetime.datetime.fromtimestamp(val)

    def set_datetime(self, key, val):
        self.set(key, time.mktime(val.timetuple()))
