#!/usr/bin/env python3

import subprocess


class BaseLogbroker:
    def list(self, path):
        output = self._run(["schema", "list", path])
        self._remove_line_with_topics_count(output)
        return [self._get_item_name(line) for line in output]

    def _run(self, cmd, comment=False):
        full_cmd = self._make_full_cmd(cmd, comment)
        output = subprocess.check_output(full_cmd).decode("utf-8").rstrip()
        return output.split("\n")

    def _make_full_cmd(self, cmd, comment):
        full_cmd = ["logbroker", "-s", "logbroker"] + cmd
        # Not all commands support -m option
        if comment:
            full_cmd += ["-m", "run by mail logctl tool"]
        return full_cmd

    def _remove_line_with_topics_count(self, cmd_output):
        cmd_output.pop()

    def _get_item_name(self, line):
        return line.split("\t")[-1]


class Logbroker(BaseLogbroker):
    def create_topic(self, topic_path):
        print('Creating topic "{}"...'.format(topic_path))
        self._run(["schema", "create", "topic", topic_path, "-p", "1", "-y"])

    def create_directory(self, path):
        print('Creating directory "{}"...'.format(path))
        self._run(["schema", "create", "directory", path, "-y"])

    def enable_yt_delivery(self, topic_path):
        print('Enabling YT delivery for topic "{}"...'.format(topic_path))
        self._run(["yt-delivery", "add", "--topic", topic_path, "--yt", "hahn", "-y"], comment=True)

    def grant_write_access(self, topic_path, tvm_id):
        print('Granting write access to tvm_id={} for topic "{}"...'.format(tvm_id, topic_path))
        self._run(
            [
                "permissions",
                "grant",
                "--path",
                topic_path,
                "--subject",
                "{}@tvm".format(tvm_id),
                "--permissions",
                "WriteTopic",
                "-y",
            ]
        )


class DryRunLogbroker(BaseLogbroker):
    def create_topic(self, topic_path):
        print('Would create topic "{}"'.format(topic_path))

    def create_directory(self, path):
        print('Would create directory "{}"'.format(path))

    def enable_yt_delivery(self, topic_path):
        print('Would enable YT delivery for topic "{}"'.format(topic_path))

    def grant_write_access(self, topic_path, tvm_id):
        print('Would grant write access to tvm_id={} for topic "{}"'.format(tvm_id, topic_path))
