import json
import os
import subprocess
from copy import deepcopy

_LOG_TEMPLATE = {
    "name": "",
    "#aspects": "@common_project:log_presets:hahn:default",
    "cluster_name": "hahn",
    "streams": [],
    "periods": [],
    "lifetimes": {},
    "build_options": "@log_options:build:common",
    "rebuild_options": "@log_options:rebuild:common",
    "archive_options": "@log_options:archive:common",
    "infer_yt_schema": True,
}

_DEFAULT_PERIODS = [{"name": "1d", "lifetime": "183d"}, {"name": "30m", "lifetime": "7d"}]

_STREAM_TEMPLATE = {
    "topic_path": "",
    "logbroker": "@stream_options:logbroker:common_hahn",
    "indexing_options": "@stream_options:indexing:schema_common",
    "parsing_options": {
        "chunk_splitter_name": "line-break",
        "parser_name": "",
        "formatter_name": "native",
    },
    "lifetimes": "@stream_options:lifetimes:common",
}

_PARSER_FIELD_TEMPLATE = {"name": "", "type": ""}

_PARSER_FIELD_TYPES = {
    "str": "VT_STRING",
    "bool": "VT_BOOLEAN",
    "int64": "VT_INT64",
    "uint64": "VT_UINT64",
    "double": "VT_DOUBLE",
    "ts": "VT_TIMESTAMP",
}

_PARSER_FILE_TEMPLATE = {"fields": []}

_PARSER_TEMPLATE = {
    "type": "TTskvWithSchemaLogParser",
    "common": {
        "time_field": "timestamp",
        "time_format_options": {"time_format": "", "time_zone_mode": "TZ_LOCAL"},
        "record_timestamp_field": "_logfeller_timestamp",
        "protect_cookie": False,
        "precompute_fields": False,
    },
    "custom": {"delimiter": "\\t", "config": ""},
    "resources": [],
}


class BaseLogfeller:
    def __init__(self):
        self.configs = {}

    def logs_config_path(self, config_name):
        return "{}/{}.json".format(self._log_configs_dir(), config_name)

    def _log_configs_dir(self):
        return "logfeller/configs/logs"

    def streams_config_path(self, config_name):
        return "{}/{}.json".format(self._stream_configs_dir(), config_name)

    def _stream_configs_dir(self):
        return self._log_configs_dir()

    def parser_config_path(self, parser_name):
        return "{}/robot_resources/{}.json".format(self._parser_configs_dir(), parser_name)

    def _parser_configs_dir(self):
        return "logfeller/configs/parsers"

    def parsers_config_path(self):
        return "{}/parsers.robot.json".format(self._parser_configs_dir())

    def parsers_make_file_path(self):
        return "{}/robot_resources/ya.make".format(self._parser_configs_dir())

    def config_exists(self, path):
        return os.path.exists(path)

    def get_logs_from_logs_config(self, path):
        config = self._read_config(path)
        return config.keys()

    def get_topics_from_streams_config(self, path):
        config = self._read_config(path)
        return config.keys()

    def get_fields_from_parser_config(self, path):
        config = self._read_config(path)
        return [self._field_str(field) for field in config["fields"]]

    def _generate_log_config(self, log_name, topic, periods=None):
        ret = deepcopy(_LOG_TEMPLATE)
        ret["name"] = log_name
        ret["streams"].append("@streams:hahn:{}".format(topic))
        if periods is None:
            periods = _DEFAULT_PERIODS
        for period in periods:
            ret["periods"].append(period["name"])
            ret["lifetimes"][period["name"]] = period["lifetime"]
        return ret

    def _generate_stream_config(self, topic, parser_name):
        ret = deepcopy(_STREAM_TEMPLATE)
        ret["topic_path"] = topic
        ret["parsing_options"]["parser_name"] = parser_name
        return ret

    def _generate_parser_config(self, fields):
        ret = deepcopy(_PARSER_FILE_TEMPLATE)
        for field in fields:
            field_data = deepcopy(_PARSER_FIELD_TEMPLATE)
            field_data["name"] = self._field_name(field)
            field_data["type"] = self._field_type(field)
            ret["fields"].append(field_data)
        return ret

    def _generate_global_parsers_config_entry(self, parser_name, time_format):
        ret = deepcopy(_PARSER_TEMPLATE)
        parser_path = "robot_resources/{}.json".format(parser_name)
        ret["common"]["time_format_options"]["time_format"] = time_format
        ret["custom"]["config"] = parser_path
        ret["resources"].append(parser_path)
        return ret

    def _field_type(self, field):
        if ":" not in field:
            return "VT_STRING"
        type_str = field.split(":")[-1]
        if type_str not in _PARSER_FIELD_TYPES:
            raise RuntimeError('Unsupported field type "{}"'.format(type_str))
        return _PARSER_FIELD_TYPES[type_str]

    def _field_name(self, field):
        return field.split(":")[0]

    def _field_str(self, field):
        if field["type"] == "VT_STRING":
            return field["name"]
        for type_str, logfeller_type_str in _PARSER_FIELD_TYPES.items():
            if field["type"] == logfeller_type_str:
                return "{}:{}".format(field["name"], type_str)
        raise RuntimeError('Unsupported field type "{}"'.format(field["type"]))

    def _write_config(self, path, content, indent=2):
        with open(path, "w") as file:
            json.dump(content, file, indent=indent, sort_keys=True)
        self.configs[path] = content

    def _read_config(self, path):
        if path in self.configs:
            return self.configs[path]
        with open(path, "r") as file:
            content = json.load(file)
            self.configs[path] = content
            return content


class Logfeller(BaseLogfeller):
    def append_to_logs_config(self, topic, log_name, path, periods=None):
        print('Appending topic "{}" to logs config...'.format(topic))
        config = self._read_config(path)
        config[log_name] = self._generate_log_config(log_name, topic, periods)
        self._write_config(path, config, indent=4)

    def append_to_streams_config(self, topic, parser_name, path):
        print('Appending topic "{}" to streams config...'.format(topic))
        config = self._read_config(path)
        config[topic] = self._generate_stream_config(topic, parser_name)
        self._write_config(path, config, indent=4)

    def create_parser_config(self, fields, path):
        print('Creating parser config: "{}"...'.format(path))
        config = self._generate_parser_config(fields)
        self._write_config(path, config)

    def rewrite_parser_config(self, fields, path):
        print('Rewriting parser config: "{}"'.format(path))
        config = self._generate_parser_config(fields)
        self._write_config(path, config)

    def append_to_parsers_config(self, parser_name, time_format):
        print('Appending parser "{}" to global parsers config...'.format(parser_name))
        parsers_config_path = self.parsers_config_path()
        parsers_config = self._read_config(parsers_config_path)
        parsers_config[parser_name] = self._generate_global_parsers_config_entry(
            parser_name, time_format
        )
        self._write_config(parsers_config_path, parsers_config)

    def update_parsers_ya_make(self):
        print("Updating parsers ya.make...")
        script_dir = "{}/robot_resources".format(self._parser_configs_dir())
        script_file = "./update_makefile.sh"
        output = subprocess.check_output(["bash", script_file], cwd=script_dir).decode("UTF-8")
        if len(output) > 0:
            print(output)


class DryRunLogfeller(BaseLogfeller):
    def append_to_logs_config(self, topic, log_name, path, periods=None):
        print('Would append topic "{}" to logs config'.format(topic))
        config = self._read_config(path)
        config[log_name] = self._generate_log_config(log_name, topic, periods)
        self._write_config(path, config)

    def append_to_streams_config(self, topic, parser_name, path):
        print('Would append topic "{}" to streams config'.format(topic))

    def create_parser_config(self, fields, path):
        print('Would create parser config: "{}"'.format(path))

    def rewrite_parser_config(self, fields, path):
        print('Would rewrite parser config: "{}"'.format(path))

    def append_to_parsers_config(self, parser_name, time_format):
        print('Would append parser "{}" to global parsers config'.format(parser_name))

    def update_parsers_ya_make(self):
        print("Would update parsers ya.make")
