import yaml
import os
from copy import deepcopy
from pathlib import Path, PurePath

_CONFIG_TEMPLATE = {
    "ident": "",
    "logger-mode": "file",
    "logger-file": "/var/log/statbox/watcher.log",
    "logger-level": 5,
    "logger-timeformat": "%d.%m.%Y-%H.%M.%S",
    "network": {
        "tvm-client-id": "",
        "tvm-server-id": "",
        "master-addr": "logbroker.yandex.net",
        "proto": "pq",
        "transport": "ipv6",
    },
    "migrate": "push",
    "files": [],
}

_FILE_CONFIG_TEMPLATE = {"name": "", "log_type": ""}

_LOGBROKER_TVM_ID = 2001059


class BasePushClient:
    def __init__(self):
        self._configs = {}

    def config_exists(self, path):
        return os.path.exists(path)

    def get_topics_from_config(self, path):
        config = self._read_config(path)
        ret = []
        for file in config["files"]:
            topic = self._get_topic(config["ident"], file["log_type"])
            ret.append(topic)
        return ret

    def _generate_log_file_config(self, topic, log_path, options=None):
        log_name = self._get_log_name(topic)
        ret = deepcopy(_FILE_CONFIG_TEMPLATE)
        ret["log_type"] = log_name  # For historical reasons field in config is named "log_type".
        ret["name"] = log_path
        if options:
            ret.update(options)
        return ret

    def _get_log_name(self, topic):
        return PurePath(topic).name

    def _get_ident(self, topic):
        directory = PurePath(topic).parent
        return str(directory).replace("/", "@")

    def _get_topic(self, ident, log):
        return "{}/{}".format(ident.replace("@", "/"), log)

    def _write_config(self, path, content):
        parent = Path(path).parent
        if not parent.exists():
            parent.mkdir(parents=True)
        with open(path, "w") as file:
            file.write(yaml.dump(content))
        self._configs[path] = content

    def _read_config(self, path):
        if path in self._configs:
            return self._configs[path]
        with open(path, "r") as file:
            content = yaml.safe_load(file)
            self._configs[path] = content
            return content


class PushClient(BasePushClient):
    def create_config(self, tvm_options, topic, log_path, config_path, options=None):
        print('Creating config "{}" with topic "{}"...'.format(config_path, topic))
        config = deepcopy(_CONFIG_TEMPLATE)
        config["ident"] = self._get_ident(topic)
        config["network"]["tvm-client-id"] = tvm_options["id"]
        config["network"]["tvm-server-id"] = _LOGBROKER_TVM_ID
        if "secret_path" in tvm_options:
            config["network"]["tvm-secret-file"] = tvm_options["secret_path"]
        else:
            config["network"]["tvm-secret"] = tvm_options["secret_value"]
        config["files"].append(self._generate_log_file_config(topic, log_path, options))
        self._write_config(config_path, config)

    def append_topic_to_config(self, topic, log_path, config_path, options=None):
        print('Adding topic "{}" to config "{}"...'.format(topic, config_path))
        config = self._read_config(config_path)
        ident = self._get_ident(topic)
        if config["ident"] != ident:
            raise RuntimeError(
                "ident mismatch: expected {} but found {}".format(ident, config["ident"])
            )
        config["files"].append(self._generate_log_file_config(topic, log_path, options))
        self._write_config(config_path, config)


class DryRunPushClient(BasePushClient):
    def create_config(self, tvm_options, topic, log_path, config_path, options=None):
        print('Would create config "{}" with topic "{}"'.format(config_path, topic))

    def append_topic_to_config(self, topic, log_path, config_path, options=None):
        print('Would add topic "{}" to config "{}"'.format(topic, config_path))
