#!/usr/bin/env python3

from lib.aux import *
from contextlib import contextmanager
from tempfile import TemporaryDirectory
from uuid import uuid4
import subprocess


class BaseArcRepository:
    def __init__(self, mount_dir, store_dir):
        self._mount_dir = mount_dir
        self._store_dir = store_dir
        self._has_commits = False

    def mount(self):
        subprocess.check_call(
            ["arc", "mount", "--mount", self._mount_dir, "--store", self._store_dir]
        )

    def unmount(self):
        subprocess.check_call(["arc", "unmount", self._mount_dir])

    def path(self):
        return self._mount_dir

    def create_branch(self, branch=None):
        if not branch:
            branch = str(uuid4())
        subprocess.check_call(["arc", "checkout", "--quiet", "-b", branch])

    def has_commits(self):
        return self._has_commits


class ArcRepository(BaseArcRepository):
    def commit(self, files, message):
        print('Commiting files {} as "{}"...'.format(files, message))
        self._add(files)
        self._commit(message)
        self._has_commits = True

    def _add(self, files):
        for file in files:
            subprocess.check_call(["arc", "add", file])

    def _commit(self, message):
        subprocess.check_call(["arc", "commit", "--message", message])

    def create_pr(self, message):
        print('Creating pr "{}"...'.format(message))
        output = subprocess.check_output(
            ["arc", "pr", "create", "--push", "--message", message]
        ).decode("utf-8")
        print(output)


class DryRunArcRepository(BaseArcRepository):
    def commit(self, files, message):
        print('Would commit files {} as "{}"'.format(files, message))
        self._has_commits = True

    def create_pr(self, message):
        print('Would create pr "{}"'.format(message))


@contextmanager
def temporary_arc_repository(path, cls=ArcRepository):
    prefix = "{}/tmp_repo_".format(path.resolve())
    with TemporaryDirectory(prefix=prefix) as mount_dir, TemporaryDirectory(
        prefix=prefix
    ) as store_dir:
        repo = cls(mount_dir, store_dir)
        repo.mount()
        try:
            yield repo
        finally:
            repo.unmount()
