#!/usr/bin/env python3

from lib.aux import *
from types import SimpleNamespace
import subprocess
import os


class ASGRepository:
    def __init__(self, asg_config: dict):
        self.config = {"asg": asg_config}
        self.cache = dict()  # type: ignore
        self.fetched = False

    def _fetch(self):
        if not self.fetched:
            self.fetched = True
            asg_path = os.path.expanduser(self.config["asg"]["path"])
            try:
                subprocess.check_output([asg_path, "noop"])
                ignored = subprocess.check_output(
                    ["git", "fetch", "--tags", "-f"], stderr=subprocess.STDOUT
                )
            except Exception as e:
                raise Exception("VCS error")

    def get_revision(self, branch: str):
        self._fetch()
        asg_path = os.path.expanduser(self.config["asg"]["path"])
        try:
            subprocess.check_output([asg_path, "noop"])
        except Exception as e:
            raise Exception("VCS error")

        try:
            # use rev-list instead of rev-parse to resolve tags to commits correctly
            return (
                subprocess.check_output(["git", "rev-list", "-n", "1", branch])
                .decode("utf-8")
                .strip()
            )
        except Exception as e:
            raise Exception("VCS error")

    def get_changelog(self, app_path: str, old_revision: str, new_revision: str):
        self._fetch()
        asg_path = os.path.expanduser(self.config["asg"]["path"])
        if old_revision == new_revision:
            return "<no changes>"

        if (old_revision, new_revision) in self.cache:
            return self.cache[(old_revision, new_revision)]

        bash_call = "{0} changelog {1} {2} {3}".format(
            asg_path, app_path, old_revision, new_revision
        )
        try:
            changelog = subprocess.check_output(bash_call.split(" ")).decode("utf-8").strip()
            self.cache[(old_revision, new_revision)] = changelog
            return changelog
        except Exception as e:
            raise Exception("VCS error")
